//
// Created by 乾三 on 2023/3/2.
//

//#include "helmetPreprocess.h"

#include <iostream>
#include <sys/timeb.h>
#include "CarParams.h"
#include "../include/detectPreprocess.h"
#include "AclLiteApp.h"
#include "../include/helmetPreprocess.h"


using namespace std;

namespace {
    uint32_t kModelWidth = 640;
    uint32_t kModelHeight = 448;
}

helmetPreprocessThread::helmetPreprocessThread(const char*& configFile,int32_t deviceId, int32_t channelId, aclrtRunMode& runMode, OnlyDetectNameInfo nameinfo, bool display,int skip)
{
    display_ = display;
    configFile_ = configFile;
    cap_ = nullptr;
    runMode_ = runMode;
    modelWidth_ = kModelWidth;
    modelHeight_ = kModelHeight;
    selfThreadId_ = INVALID_INSTANCE_ID;
    inferThreadId_ = INVALID_INSTANCE_ID;
    detectPostThreadId_ = INVALID_INSTANCE_ID;
    presentAgentDisplayThreadId_ = INVALID_INSTANCE_ID;
    rtspDisplayThreadId_ = INVALID_INSTANCE_ID;
    frameCnt_ = 0;
    deviceId_ = deviceId;
    channelId_ = channelId;
    kDetectPostName_ = nameinfo.kDetectPostName;
    kInferName_ = nameinfo.kInferName;
    skipframe_ = skip;
    krtspurl_ = nameinfo.kRtspUrl;
    kMqttSendName_ = nameinfo.kMqttName;

}

helmetPreprocessThread::~helmetPreprocessThread()
{

}



AclLiteError helmetPreprocessThread::Init()
{
//    std::map<std::string, std::string> config;
//    if (!ReadConfig(config, configFile_)) {
//        return ACLLITE_ERROR;
//    }

    AclLiteError aclRet = GetInputDataType(inputType_, channelId_);
    if (aclRet != ACLLITE_OK) {
        ACLLITE_LOG_ERROR("GetInputDataType failed, error %d", aclRet);
        return ACLLITE_ERROR;
    }
    skipframe_ = 0;
    aclRet = GetSkip(skipframe_, channelId_);
    if (aclRet != ACLLITE_OK) {
        ACLLITE_LOG_ERROR("Getskipframe_ failed, error %d", aclRet);
        return ACLLITE_ERROR;
    }

    aclRet = GetInputDataPath(inputDataPath_, channelId_);
    if (aclRet != ACLLITE_OK) {
        ACLLITE_LOG_ERROR("GetInputDataPath failed, error %d", aclRet);
        return ACLLITE_ERROR;
    }



    if (inputType_ == "pic") {
        aclRet = OpenPicsDir();
    } else {
        aclRet = OpenVideoCapture();
    }
    if (aclRet != ACLLITE_OK) {
        return ACLLITE_ERROR;
    }

    aclRet = dvpp_.Init();
    if (aclRet) {
        ACLLITE_LOG_ERROR("Dvpp init failed, error %d", aclRet);
        return ACLLITE_ERROR;
    }

    // Get the relevant thread instance id
    selfThreadId_ = SelfInstanceId();
    inferThreadId_ = GetAclLiteThreadIdByName(kInferName_ + to_string(deviceId_));
    detectPostThreadId_ = GetAclLiteThreadIdByName(kDetectPostName_ + to_string(channelId_));
    classifyPreThreadId_ = -1;
    classifyPostThreadId_ = -1;
    mqttSendThreadId_ = GetAclLiteThreadIdByName(kMqttSendName_ + to_string(channelId_));
    rtspDisplayThreadId_ = GetAclLiteThreadIdByName(krtspurl_ + to_string(channelId_));
    if (display_) {
        presentAgentDisplayThreadId_ = GetAclLiteThreadIdByName(kPresentAgentDisplayName.c_str());
    }
    if ((selfThreadId_ == INVALID_INSTANCE_ID) ||
        (inferThreadId_ == INVALID_INSTANCE_ID) ||
        (detectPostThreadId_ == INVALID_INSTANCE_ID)
        ) {
        ACLLITE_LOG_ERROR("Self instance id %d, infer instance id %d, detectPost instance id %d,"
                          "...",
                          selfThreadId_, inferThreadId_, detectPostThreadId_
                          );
        return ACLLITE_ERROR;
    }

    return ACLLITE_OK;
}

AclLiteError helmetPreprocessThread::GetInputDataType(std::string& inputType, uint32_t channelId)
{
    std::string inputTypeKey = "inputType_" + to_string(channelId);
    std::map<std::string, std::string> config;
    if (!ReadConfig(config, configFile_)) {
        return ACLLITE_ERROR;
    }

    std::map<std::string, std::string>::const_iterator mIter = config.begin();
    for (; mIter != config.end(); ++mIter) {
        if (mIter->first == inputTypeKey) {
            inputType.assign(mIter->second.c_str());
            ACLLITE_LOG_INFO("device %d input type is : %s",
                             channelId, inputType.c_str());
        }
    }
    if (inputType.empty() || (inputType != "video" &&
                              inputType != "pic" && inputType != "rtsp")) {
        ACLLITE_LOG_ERROR("device %d input type is invalid", channelId);
        return ACLLITE_ERROR;
    }

    return ACLLITE_OK;
}

AclLiteError helmetPreprocessThread::GetSkip(int& skip, uint32_t channelId)
{
    std::string skipKey = "skip_" + to_string(channelId);
    std::map<std::string, std::string> config;
    if (!ReadConfig(config, configFile_)) {
        return ACLLITE_ERROR;
    }
    std::string skipResult;
    std::map<std::string, std::string>::const_iterator mIter = config.begin();
    for (; mIter != config.end(); ++mIter) {
        if (mIter->first == skipKey) {
            skipResult.assign(mIter->second.c_str());
            ACLLITE_LOG_INFO("device %d skipKey num is : %s",
                             channelId, skipResult.c_str());
        }
    }
    skip = std::stoi(skipResult);
    if (skipResult.empty()) {
        ACLLITE_LOG_ERROR("device %d skipKey num is invalid", channelId);
        return ACLLITE_ERROR;
    }


    return ACLLITE_OK;
}

AclLiteError helmetPreprocessThread::Process(int msgId, shared_ptr<void> msgData)
{
    AclLiteError ret = ACLLITE_OK;
    shared_ptr<ObjDetectDataMsg> objDetectDataMsg = make_shared<ObjDetectDataMsg>();
    switch (msgId) {
        case MSG_APP_START:
            ret = AppStart();
            break;
        case MSG_READ_FRAME:
            ret = MsgRead(objDetectDataMsg);
            ret = MsgProcess(objDetectDataMsg);
            ret = MsgSend(objDetectDataMsg);
            break;
        default:
            ACLLITE_LOG_ERROR("Detect Preprocess thread receive unknow msg %d", msgId);
            break;
    }

    return ret;
}

AclLiteError helmetPreprocessThread::MsgSend(shared_ptr<ObjDetectDataMsg> &objDetectDataMsg)
{
    AclLiteError ret;
    if (objDetectDataMsg->isLastFrame == 0){
        while (1) {
            ret = SendMessage(objDetectDataMsg->inferThreadId, MSG_DETECT_PREPROC_DATA, objDetectDataMsg);
            if (ret == ACLLITE_ERROR_ENQUEUE) {
                usleep(500);
                continue;
            } else if(ret == ACLLITE_OK) {
                break;
            } else {
                ACLLITE_LOG_ERROR("Send read frame message failed, error %d", ret);
                return ret;
            }
        }

        ret = SendMessage(selfThreadId_, MSG_READ_FRAME, nullptr);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Send read frame message failed, error %d", ret);
            return ret;
        }
    } else {
        shared_ptr<ObjDetectDataMsg> objDetectDataMsgEnd = make_shared<ObjDetectDataMsg>();
        objDetectDataMsgEnd->inferThreadId = inferThreadId_;
        objDetectDataMsgEnd->detectPostThreadId = detectPostThreadId_;
        objDetectDataMsgEnd->classifyPreThreadId = classifyPreThreadId_;
        objDetectDataMsgEnd->classifyPostThreadId = classifyPostThreadId_;
        objDetectDataMsgEnd->presentAgentDisplayThreadId = presentAgentDisplayThreadId_;
        objDetectDataMsgEnd->rtspDisplayThreadId = rtspDisplayThreadId_;
        objDetectDataMsgEnd->mqttSendThreadId = mqttSendThreadId_;
        objDetectDataMsgEnd->deviceId = deviceId_;
        objDetectDataMsgEnd->channelId = channelId_;
        objDetectDataMsgEnd->frameNum = objDetectDataMsg->frameNum;
        objDetectDataMsgEnd->isLastFrame = objDetectDataMsg->isLastFrame;
        while (1) {
            ret = SendMessage(objDetectDataMsgEnd->inferThreadId, MSG_DETECT_PREPROC_DATA, objDetectDataMsgEnd);
            if (ret == ACLLITE_ERROR_ENQUEUE) {
                usleep(500);
                continue;
            } else if(ret == ACLLITE_OK) {
                break;
            } else {
                ACLLITE_LOG_ERROR("Send read frame message failed, error %d", ret);
                return ret;
            }
        }
    }
    return ACLLITE_OK;
}

AclLiteError helmetPreprocessThread::GetInputDataPath(std::string& inputDataPath, uint32_t channelId)
{
    std::string inputDataPathKey = "inputDataPath_" + to_string(channelId);
    std::map<std::string, std::string> config;
    if (!ReadConfig(config, configFile_)) {
        return ACLLITE_ERROR;
    }

    std::map<std::string, std::string>::const_iterator mIter = config.begin();
    for (; mIter != config.end(); ++mIter) {
        if (mIter->first == inputDataPathKey) {
            inputDataPath.assign(mIter->second.c_str());
            ACLLITE_LOG_INFO("device %d input data path is : %s",
                             channelId, inputDataPath.c_str());
        }
    }
    if (inputDataPath.empty()) {
        ACLLITE_LOG_ERROR("device %d input data path is invalid", channelId);
        return ACLLITE_ERROR;
    }

    return ACLLITE_OK;
}

AclLiteError helmetPreprocessThread::OpenPicsDir()
{
    string inputImageDir = inputDataPath_;
    GetAllFiles(inputImageDir, fileVec_);
    if (fileVec_.empty()) {
        ACLLITE_LOG_ERROR("Failed to deal all empty path=%s.", inputImageDir.c_str());
        return ACLLITE_ERROR;
    }

    return ACLLITE_OK;
}

AclLiteError helmetPreprocessThread::OpenVideoCapture()
{
//    inputDataPath_ = inputDataPath_.substr(0,inputDataPath_.length()-1);
    if (IsRtspAddr(inputDataPath_)) {
        cap_ = new AclLiteVideoProc(inputDataPath_, deviceId_);
    } else if (IsVideoFile(inputDataPath_)) {
        if (!IsPathExist(inputDataPath_)) {
            ACLLITE_LOG_ERROR("The %s is inaccessible", inputDataPath_.c_str());
            return ACLLITE_ERROR;
        }
        cap_ = new AclLiteVideoProc(inputDataPath_, deviceId_);
    } else {
        ACLLITE_LOG_ERROR("Invalid param. The arg should be accessible rtsp,"
                          " video file or camera id");
        return ACLLITE_ERROR;
    }

    if (!cap_->IsOpened()) {
        delete cap_;
        ACLLITE_LOG_ERROR("Failed to open video");
        return ACLLITE_ERROR;
    }

    return ACLLITE_OK;
}

AclLiteError helmetPreprocessThread::AppStart()
{
    AclLiteError ret = SendMessage(selfThreadId_, MSG_READ_FRAME, nullptr);
    if (ret != ACLLITE_OK) {
        ACLLITE_LOG_ERROR("Process app start message failed, error %d", ret);
    }

    return ret;
}

AclLiteError helmetPreprocessThread::MsgRead(shared_ptr<ObjDetectDataMsg> &objDetectDataMsg)
{
    AclLiteError ret;
    if (inputType_ == "pic") {
        ret = ReadPic(objDetectDataMsg);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Read pic failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    } else {
        ret = ReadStream(objDetectDataMsg);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Read frame failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    }

    return ACLLITE_OK;
}

AclLiteError helmetPreprocessThread::MsgProcess(shared_ptr<ObjDetectDataMsg> &objDetectDataMsg)
{
    AclLiteError ret;
    if (inputType_ == "pic") {
        ret = ProcessPic(objDetectDataMsg);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Process pic failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    } else {
        ret = ProcessStreamFrame(objDetectDataMsg);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Process stream frame failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    }

    return ACLLITE_OK;
}


AclLiteError helmetPreprocessThread::ReadStream(shared_ptr<ObjDetectDataMsg> &objDetectDataMsg)
{
    objDetectDataMsg->inferThreadId = inferThreadId_;
    objDetectDataMsg->detectPostThreadId = detectPostThreadId_;
    objDetectDataMsg->classifyPreThreadId = classifyPreThreadId_;
    objDetectDataMsg->classifyPostThreadId = classifyPostThreadId_;
    objDetectDataMsg->mqttSendThreadId = mqttSendThreadId_;
    objDetectDataMsg->presentAgentDisplayThreadId = presentAgentDisplayThreadId_;
    objDetectDataMsg->rtspDisplayThreadId = rtspDisplayThreadId_;
    objDetectDataMsg->deviceId = deviceId_;
    objDetectDataMsg->channelId = channelId_;
    objDetectDataMsg->frameNum = frameCnt_;
    objDetectDataMsg->isLastFrame = 0;
    ImageData skipImageFrame;
    AclLiteError ret;
    int index = 0;
    if (skipframe_>1){
        while(1) {

            ret = cap_->Read(skipImageFrame);

            if (ret != ACLLITE_OK) {
                if (ret == ACLLITE_ERROR_DECODE_FINISH) {
                    objDetectDataMsg->isLastFrame = 1;
                    return ACLLITE_OK;
                } else {
                    objDetectDataMsg->isLastFrame = 1;
                    return ret;
                }
            }
            index++;
            if (index >skipframe_)
                break;

        }
    }

    ret = cap_->Read(objDetectDataMsg->imageFrame);
    if (ret != ACLLITE_OK) {
        if (ret == ACLLITE_ERROR_DECODE_FINISH) {
            objDetectDataMsg->isLastFrame = 1;
            return ACLLITE_OK;
        } else {
            objDetectDataMsg->isLastFrame = 1;
            return ret;
        }
    }
    frameCnt_++;
    return ACLLITE_OK;
}
//AclLiteError helmetPreprocessThread::ReadPic(shared_ptr<ObjDetectDataMsg> &objDetectDataMsg)
//{
//    objDetectDataMsg->inferThreadId = inferThreadId_;
//    objDetectDataMsg->detectPostThreadId = detectPostThreadId_;
//    objDetectDataMsg->classifyPreThreadId = classifyPreThreadId_;
//    objDetectDataMsg->classifyPostThreadId = classifyPostThreadId_;
//    objDetectDataMsg->presentAgentDisplayThreadId = presentAgentDisplayThreadId_;
//    objDetectDataMsg->rtspDisplayThreadId = rtspDisplayThreadId_;
//    objDetectDataMsg->deviceId = deviceId_;
//    objDetectDataMsg->channelId = channelId_;
//    objDetectDataMsg->frameNum = frameCnt_;
//    objDetectDataMsg->isLastFrame = 0;
//
//    if (frameCnt_ == fileVec_.size()) {
//        objDetectDataMsg->isLastFrame = 1;
//        return ACLLITE_OK;
//    }
//    string picFile = fileVec_[frameCnt_];
//    AclLiteError ret = ReadJpeg(objDetectDataMsg->imageFrame, picFile);
//    objDetectDataMsg->frame = cv::imread(picFile);
//    frameCnt_++;
//    return ACLLITE_OK;
//}


