/**
* Copyright (c) Huawei Technologies Co., Ltd. 2020-2022. All rights reserved.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at

* http://www.apache.org/licenses/LICENSE-2.0

* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.

* File sample_process.cpp
* Description: handle acl resource
*/
#include <iostream>
#include "acl/acl.h"
#include "CarParams.h"
#include "../include/detectPostprocess.h"
#include "AclLiteUtils.h"
#include "AclLiteApp.h"

using namespace std;

namespace {
const uint32_t kBBoxDataBufId = 0;
const uint32_t kBoxNumDataBufId = 1;

enum BBoxIndex { TOPLEFTX = 0, TOPLEFTY = 1, BOTTOMRIGHTX = 2, BOTTOMRIGHTY = 3, SCORE = 4, LABEL = 5 };
//uint32_t kModelWidth = 640;
//uint32_t kModelHeight = 448;
}

DetectPostprocessThread::DetectPostprocessThread() {
}

DetectPostprocessThread::~DetectPostprocessThread() {
}

AclLiteError DetectPostprocessThread::Init()
{
    AclLiteError ret = GetThreshold(threshold_, channelId_);
    if (ret != ACLLITE_OK) {
        return ACLLITE_ERROR;
    }
    return ACLLITE_OK;
}

AclLiteError DetectPostprocessThread::GetThreshold(float& threshold, uint32_t channelId)
{
    std::string thresholdKey = "threshold_" + to_string(channelId);
    std::map<std::string, std::string> config;
    if (!ReadConfig(config, configFile_)) {
        return ACLLITE_ERROR;
    }
    std::string thresholdResult;
    std::map<std::string, std::string>::const_iterator mIter = config.begin();
    for (; mIter != config.end(); ++mIter) {
        if (mIter->first == thresholdKey) {
            thresholdResult.assign(mIter->second.c_str());
            ACLLITE_LOG_INFO("device %d thresholdResult  is : %s",
                             channelId, thresholdResult.c_str());
        }
    }
    threshold = std::stof(thresholdResult);

    return ACLLITE_OK;
}

AclLiteError DetectPostprocessThread::Process(int msgId, shared_ptr<void> data)
{
    AclLiteError ret = ACLLITE_OK;
    switch (msgId) {
        case MSG_DETECT_INFER_OUTPUT:
            InferOutputProcess(static_pointer_cast<ObjDetectDataMsg>(data));
            MsgSend(static_pointer_cast<ObjDetectDataMsg>(data));
            break;
        default:
            ACLLITE_LOG_INFO("Detect PostprocessThread thread ignore msg %d", msgId);
            break;
    }

    return ret;
}

AclLiteError DetectPostprocessThread::MsgSend(shared_ptr<ObjDetectDataMsg> objDetectDataMsg)
{
    while (1) {
        AclLiteError ret = SendMessage(objDetectDataMsg->classifyPreThreadId,
            MSG_DETECT_POSTPROC_DATA, objDetectDataMsg);
        if (ret == ACLLITE_ERROR_ENQUEUE) {
            usleep(500);
            continue;
        } else if (ret == ACLLITE_OK) {
            break;
        } else {
            ACLLITE_LOG_ERROR("Send read frame message failed, error %d", ret);
            return ret;
        }
    }

    return ACLLITE_OK;
}

vector<BBoxstr> DetectPostprocessThread::nonMaximumSuppression(const float nmsThresh, std::vector<BBoxstr> binfo)
{
    auto overlap1D = [](float x1min, float x1max, float x2min, float x2max) -> float {
        float left = max(x1min, x2min);
        float right = min(x1max, x2max);
        return right-left;
    };
    auto computeIoU =[&overlap1D](BBoxstr& bbox1, BBoxstr& bbox2) -> float {
        float overlapX = overlap1D(bbox1.rect.ltX, bbox1.rect.rbX, bbox2.rect.ltX, bbox2.rect.rbX);
        float overlapY = overlap1D(bbox1.rect.ltY, bbox1.rect.rbY, bbox2.rect.ltY, bbox2.rect.rbY);
        if(overlapX <= 0 or overlapY <= 0) return 0;
        float area1 = (bbox1.rect.rbX - bbox1.rect.ltX) * (bbox1.rect.rbY - bbox1.rect.ltY);
        float area2 = (bbox2.rect.rbX - bbox2.rect.ltX) * (bbox2.rect.rbY - bbox2.rect.ltY);
        float overlap2D = overlapX * overlapY;
        float u = area1 + area2 - overlap2D;
        return u == 0 ? 0 : overlap2D / u;
    };

    std::stable_sort(binfo.begin(), binfo.end(),
                     [](const BBoxstr& b1, const BBoxstr& b2) { return b1.score > b2.score;});
    std::vector<BBoxstr> out;


    for (auto& i : binfo)
    {
        bool keep = true;
        for (auto& j : out)
        {
            if (keep)
            {
                float overlap = computeIoU(i, j);
                keep = overlap <= nmsThresh;
            }
            else
                break;
        }
        if (keep) out.push_back(i);
    }
    return out;
}

vector<BBoxstr> DetectPostprocessThread::nmsAllClasses(const float nmsThresh, std::vector<BBoxstr>& binfo, const uint numClasses)
{
    std::vector<BBoxstr> result;
    std::vector<std::vector<BBoxstr>> splitBoxes(numClasses);
    for (auto& box : binfo)
    {
        splitBoxes.at(box.cls).push_back(box);
    }

    for (auto& boxes : splitBoxes)
    {
        boxes = nonMaximumSuppression(nmsThresh, boxes);
        result.insert(result.end(), boxes.begin(), boxes.end());
    }

    return result;
}

vector<BBoxstr> DetectPostprocessThread::outPutDecode(float* detectData,ModelInfo modelinfo, int image_ori_width,int image_ori_height,int image_resize_width,int image_resize_height){

    vector<BBoxstr> detectResults;
//    cocoInfo cocoinfo;
    for (uint32_t i = 0; i < totalBox_; i++) {
        uint32_t  lineSize = modelinfo.classnum + 5;
        float score = detectData[i * lineSize + SCORE];
        if (score < threshold_) {
            continue;
        }
        BBoxstr boundBox;


        float widthScale =  float(image_resize_width) / float(image_ori_width);
        float heightScale = float(image_resize_height) / float(image_ori_height);

        int center_x = uint32_t(detectData[i * lineSize + TOPLEFTX]);
        int center_y = uint32_t(detectData[i * lineSize + TOPLEFTY]);
        int width = uint32_t(detectData[i * lineSize + BOTTOMRIGHTX]);
        int height = uint32_t(detectData[i * lineSize +  BOTTOMRIGHTY]);

        if (heightScale > widthScale)
        {
            boundBox.rect.ltX = std::max((int)((center_x - width / 2) / widthScale),1);
            boundBox.rect.rbX = std::min((int)((center_x + width / 2) / widthScale),image_ori_width) ;
            boundBox.rect.ltY = std::max((int)((((center_y - height / 2) - (image_resize_height - widthScale * image_ori_height) / 2)) / widthScale),1);
            boundBox.rect.rbY = std::min((int)((((center_y + height / 2) - (image_resize_height - widthScale * image_ori_height) / 2)) / widthScale),image_ori_height-1);

        }
        else
        {
            boundBox.rect.ltX = std::max((int)(((center_x - width / 2) - (image_resize_width - heightScale * image_ori_width) / 2) / heightScale),1);
            boundBox.rect.rbX = std::min((int)( ((center_x + width / 2) - (image_resize_width - heightScale * image_ori_width) / 2) / heightScale),image_ori_width-1);
            boundBox.rect.ltY = std::max((int)((center_y - height / 2) / heightScale),0);
            boundBox.rect.rbY = std::min((int)((center_y + height / 2) / heightScale),image_ori_height);
        }


        float Maxclass = 0.0f;
        uint32_t Maxclass_Loc = -1;
        for (int j = 0; j < modelinfo.classnum; ++j)
        {
            float class_prob = detectData[i * lineSize + LABEL + j];
            if (Maxclass < class_prob)
            {
                Maxclass = class_prob;
                Maxclass_Loc = j;
            }
        }
        boundBox.score = Maxclass * score;
        if (boundBox.score < threshold_){ // 这里再次过滤阈值 是因为 。上面那个 socre 为 此处anchor 是否是前背景的score 此处的score 为对每个类别的置信度 计算以后的 score
            continue;
        }
        boundBox.cls = Maxclass_Loc;
        detectResults.emplace_back(boundBox);
    }

    std::vector<BBoxstr> bboxesNew = nmsAllClasses(modelinfo.nmsThresh, detectResults, modelinfo.classnum);
    return bboxesNew;
}

AclLiteError DetectPostprocessThread::InferOutputProcess(shared_ptr<ObjDetectDataMsg> objDetectDataMsg)
{
    if (objDetectDataMsg->isLastFrame)
        return ACLLITE_OK;

    float* detectData = (float *)objDetectDataMsg->detectInferData[kBBoxDataBufId].data.get();
    if (detectData == nullptr) {
        ACLLITE_LOG_ERROR("detect inferoutput is null\n");
        return ACLLITE_ERROR;
    }

    totalBox_ = modelInfo_.totalBox;
    std::vector<BBoxstr> bboxesNew = outPutDecode(detectData, modelInfo_, objDetectDataMsg->imageFrame.width,objDetectDataMsg->imageFrame.height,objDetectDataMsg->resizedFrame.alignWidth,
                                                  objDetectDataMsg->resizedFrame.alignHeight);
    objDetectDataMsg->objInfo = static_cast<const vector<ObjInfo>>(NULL);
    for (auto& bboxesNew_i : bboxesNew)
    {
        ObjInfo objInfo;
        objInfo.rectangle.lt.x = bboxesNew_i.rect.ltX;
        objInfo.rectangle.lt.y = bboxesNew_i.rect.ltY;
        objInfo.rectangle.rb.x = bboxesNew_i.rect.rbX;
        objInfo.rectangle.rb.y = bboxesNew_i.rect.rbY;
        if (bboxesNew_i.cls == 0)
        {
            auto constr = std::to_string(bboxesNew_i.score);
            constr =  constr.substr(0, constr.find(".") + 3);
            objInfo.detect_result = modelInfo_.Label[bboxesNew_i.cls]  + '_'+ constr;
            objDetectDataMsg->objInfo.emplace_back(objInfo);
        }

    }
    return ACLLITE_OK;
}

//DetectPostprocessThread::DetectPostprocessThread(ModelInfo programinfo)
//        :modelInfo_(programinfo)
//{
//}
DetectPostprocessThread::DetectPostprocessThread(const char*& configFile, int channelId,ModelInfo programinfo)
        :configFile_(configFile), channelId_(channelId), modelInfo_(programinfo)
{
}






onlyDetectPostprocessThread::onlyDetectPostprocessThread() {
}

onlyDetectPostprocessThread::~onlyDetectPostprocessThread() {
    configFile_ = nullptr;
}

AclLiteError onlyDetectPostprocessThread::Init()
{
    AclLiteError ret = GetOutputDataType(outputType_, channelId_);
    if (ret != ACLLITE_OK) {
        return ACLLITE_ERROR;
    }

    ret = GetThreshold(threshold_, channelId_);
    if (ret != ACLLITE_OK) {
        return ACLLITE_ERROR;
    }

    if (outputType_ == "video") {
        ret = GetOutputFrameResolution(outputFrameWidth_, outputFrameHeight_, channelId_);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Set output frame resolution failed, error %d", ret);
            return ACLLITE_ERROR;
        }
        ret = SetOutputVideo();
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("SetOutputVideo failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    }

    return ret;
}

AclLiteError onlyDetectPostprocessThread::Process(int msgId, shared_ptr<void> data)
{

    AclLiteError ret = ACLLITE_OK;
    switch (msgId) {
        case MSG_DETECT_INFER_OUTPUT:
            InferOutputProcess(static_pointer_cast<ObjDetectDataMsg>(data));
            break;
        case MSG_ENCODE_FINISH:
            SendMessage(g_MainThreadId, MSG_APP_EXIT, nullptr);
            break;
        default:
            ACLLITE_LOG_INFO("only detect  Postprocess thread ignore msg %d", msgId);
            break;
    }

    return ret;
}

AclLiteError onlyDetectPostprocessThread::GetOutputDataType(std::string& outputType, uint32_t channelId)
{
    std::string outputTypeKey = "outputType_" + to_string(channelId);
    std::map<std::string, std::string> config;
    if (!ReadConfig(config, configFile_)) {
        return ACLLITE_ERROR;
    }

    std::map<std::string, std::string>::const_iterator mIter = config.begin();
    for (; mIter != config.end(); ++mIter) {
        if (mIter->first == outputTypeKey) {
            outputType.assign(mIter->second.c_str());
            ACLLITE_LOG_INFO("device %d output type is : %s",
                             channelId, outputType.c_str());
        }
    }
    if (outputType.empty() || (outputType != "video" &&
                               outputType != "pic" && outputType != "presentagent" &&
                               outputType != "stdout" && outputType != "rtsp" &&
                               outputType != "mqtt"
                               )) {
        ACLLITE_LOG_ERROR("device %d output type is invalid", channelId);
        return ACLLITE_ERROR;
    }

    return ACLLITE_OK;
}

AclLiteError onlyDetectPostprocessThread::GetThreshold(float& threshold, uint32_t channelId)
{
    std::string thresholdKey = "threshold_" + to_string(channelId);
    std::map<std::string, std::string> config;
    if (!ReadConfig(config, configFile_)) {
        return ACLLITE_ERROR;
    }
    std::string thresholdResult;
    std::map<std::string, std::string>::const_iterator mIter = config.begin();
    for (; mIter != config.end(); ++mIter) {
        if (mIter->first == thresholdKey) {
            thresholdResult.assign(mIter->second.c_str());
            ACLLITE_LOG_INFO("device %d thresholdResult  is : %s",
                             channelId, thresholdResult.c_str());
        }
    }
    threshold = std::stof(thresholdResult);

    return ACLLITE_OK;
}

vector<BBoxstr> onlyDetectPostprocessThread::nonMaximumSuppression(const float nmsThresh, std::vector<BBoxstr> binfo)
{
    auto overlap1D = [](float x1min, float x1max, float x2min, float x2max) -> float {
        float left = max(x1min, x2min);
        float right = min(x1max, x2max);
        return right-left;
    };
    auto computeIoU =[&overlap1D](BBoxstr& bbox1, BBoxstr& bbox2) -> float {
        float overlapX = overlap1D(bbox1.rect.ltX, bbox1.rect.rbX, bbox2.rect.ltX, bbox2.rect.rbX);
        float overlapY = overlap1D(bbox1.rect.ltY, bbox1.rect.rbY, bbox2.rect.ltY, bbox2.rect.rbY);
        if(overlapX <= 0 or overlapY <= 0) return 0;
        float area1 = (bbox1.rect.rbX - bbox1.rect.ltX) * (bbox1.rect.rbY - bbox1.rect.ltY);
        float area2 = (bbox2.rect.rbX - bbox2.rect.ltX) * (bbox2.rect.rbY - bbox2.rect.ltY);
        float overlap2D = overlapX * overlapY;
        float u = area1 + area2 - overlap2D;
        return u == 0 ? 0 : overlap2D / u;
    };

    std::stable_sort(binfo.begin(), binfo.end(),
                     [](const BBoxstr& b1, const BBoxstr& b2) { return b1.score > b2.score;});
    std::vector<BBoxstr> out;


    for (auto& i : binfo)
    {
        bool keep = true;
        for (auto& j : out)
        {
            if (keep)
            {
                float overlap = computeIoU(i, j);
                keep = overlap <= nmsThresh;
            }
            else
                break;
        }
        if (keep) out.push_back(i);
    }
    return out;
}

vector<BBoxstr> onlyDetectPostprocessThread::nmsAllClasses(const float nmsThresh, std::vector<BBoxstr>& binfo, const uint numClasses)
{
    std::vector<BBoxstr> result;
    std::vector<std::vector<BBoxstr>> splitBoxes(numClasses);
    for (auto& box : binfo)
    {
        splitBoxes.at(box.cls).push_back(box);
    }

    for (auto& boxes : splitBoxes)
    {
        boxes = nonMaximumSuppression(nmsThresh, boxes);
        result.insert(result.end(), boxes.begin(), boxes.end());
    }

    return result;
}

vector<BBoxstr> onlyDetectPostprocessThread::outPutDecode(float* detectData,ModelInfo modelinfo, int image_ori_width,int image_ori_height,int image_resize_width,int image_resize_height){

    vector<BBoxstr> detectResults;
//    cocoInfo cocoinfo;
    for (uint32_t i = 0; i < totalBox_; i++) {
        uint32_t  lineSize = modelinfo.classnum + 5;
        float score = detectData[i * lineSize + SCORE];
        if (score < threshold_) {
            continue;
        }
        BBoxstr boundBox;


        float widthScale =  float(image_resize_width) / float(image_ori_width);
        float heightScale = float(image_resize_height) / float(image_ori_height);

        int center_x = uint32_t(detectData[i * lineSize + TOPLEFTX]);
        int center_y = uint32_t(detectData[i * lineSize + TOPLEFTY]);
        int width = uint32_t(detectData[i * lineSize + BOTTOMRIGHTX]);
        int height = uint32_t(detectData[i * lineSize +  BOTTOMRIGHTY]);

        if (heightScale > widthScale)
        {
            boundBox.rect.ltX = std::max((center_x - width / 2),0) / widthScale;
            boundBox.rect.rbX = std::min((center_x + width / 2),image_ori_width) / widthScale;
            boundBox.rect.ltY = std::max((int)((center_y - height / 2) - (image_resize_height - widthScale * image_ori_height) / 2),0) / widthScale;
            boundBox.rect.rbY = std::max((int)((center_y + height / 2) - (image_resize_height - widthScale * image_ori_height) / 2),0) / widthScale;

        }
        else
        {
            boundBox.rect.ltX = std::max((int)((center_x - width / 2) - (image_resize_width - heightScale * image_ori_width) / 2),0) / heightScale;
            boundBox.rect.rbX = std::max((int)((center_x + width / 2) - (image_resize_width - heightScale * image_ori_width) / 2),0) / heightScale;
            boundBox.rect.ltY = std::max((center_y - height / 2),0) / heightScale;
            boundBox.rect.rbY = std::min((center_y + height / 2),image_ori_height) / heightScale;
        }



        float Maxclass = 0.0f;
        uint32_t Maxclass_Loc = -1;
        for (int j = 0; j < modelinfo.classnum; ++j)
        {
            float class_prob = detectData[i * lineSize + LABEL + j];
            if (Maxclass < class_prob)
            {
                Maxclass = class_prob;
                Maxclass_Loc = j;
            }
        }
        boundBox.score = Maxclass * score;
        if (boundBox.score < threshold_){ // 这里再次过滤阈值 是因为 。上面那个 socre 为 此处anchor 是否是前背景的score 此处的score 为对每个类别的置信度 计算以后的 score
            continue;
        }
        boundBox.cls = Maxclass_Loc;
        detectResults.emplace_back(boundBox);
    }

    std::vector<BBoxstr> bboxesNew = nmsAllClasses(modelinfo.nmsThresh, detectResults, modelinfo.classnum);
    return bboxesNew;
}

AclLiteError onlyDetectPostprocessThread::InferOutputProcess(shared_ptr<ObjDetectDataMsg> objDetectDataMsg)
{
    if (objDetectDataMsg->isLastFrame == 1) {
        if (outputType_ == "video") {
            outputVideo_.release();
        }
        if (outputType_ != "presentagent" && outputType_ != "rtsp" && outputType_ != "mqtt") {

            SendMessage(objDetectDataMsg->detectPostThreadId, MSG_ENCODE_FINISH, nullptr);
            ACLLITE_LOG_INFO("it is lastframe in only detect Post without presentagent");
            return ACLLITE_OK;
        } else if (outputType_ == "pic") {
            stringstream sstream;
            sstream.str("");
            sstream << "../out/output/device_" << objDetectDataMsg->deviceId
                    << "_out_pic_" << objDetectDataMsg->frameNum << ".jpg";
            cv::imwrite(sstream.str(), objDetectDataMsg->frame);
            return ACLLITE_OK;
        } else {
            AclLiteError ret = DisplayMsgSend(objDetectDataMsg);
            if (ret != ACLLITE_OK) {
                ACLLITE_LOG_ERROR("Send last msg in classifyPost failed, error %d", (int)ret);
                return ACLLITE_ERROR;
            }
            ACLLITE_LOG_INFO("it is lastframe in classifyPost with presentagent/rtsp");
            return ACLLITE_OK;
        }
    }

    float* detectData = (float *)objDetectDataMsg->detectInferData[kBBoxDataBufId].data.get();
    if (detectData == nullptr) {
        ACLLITE_LOG_ERROR("detect inferoutput is null\n");
        return ACLLITE_ERROR;
    }

    totalBox_ = modelInfo_.totalBox;
    std::vector<BBoxstr> bboxesNew = outPutDecode(detectData, modelInfo_, objDetectDataMsg->imageFrame.width,objDetectDataMsg->imageFrame.height,objDetectDataMsg->resizedFrame.alignWidth,
                                                  objDetectDataMsg->resizedFrame.alignHeight);
    for (auto& bboxesNew_i : bboxesNew)
    {
        ObjInfo objInfo;
        objInfo.rectangle.lt.x = bboxesNew_i.rect.ltX;
        objInfo.rectangle.lt.y = bboxesNew_i.rect.ltY;
        objInfo.rectangle.rb.x = bboxesNew_i.rect.rbX;
        objInfo.rectangle.rb.y = bboxesNew_i.rect.rbY;
//        if (bboxesNew_i.cls == 2)

//        {
        if (modelInfo_.activateLabel.empty()){

            auto constr = std::to_string(bboxesNew_i.score);
            constr =  constr.substr(0, constr.find(".") + 3);
            objInfo.detect_result = modelInfo_.Label[bboxesNew_i.cls]  + '_'+ constr;
            objDetectDataMsg->objInfo.emplace_back(objInfo);

        } else{

            for(int v_i = 0 ;v_i < modelInfo_.activateLabel.size(); v_i++)
            {
                if (modelInfo_.activateLabel[v_i] == bboxesNew_i.cls){

                    auto constr = std::to_string(bboxesNew_i.score);
                    constr =  constr.substr(0, constr.find(".") + 3);
                    objInfo.detect_result = modelInfo_.Label[bboxesNew_i.cls]  + '_'+ constr;
                    objDetectDataMsg->objInfo.emplace_back(objInfo);
                    break;
                }

            }
        }



    }

    AclLiteError ret;
    if (outputType_ == "video") {
        ret = DrawResultOnVideo(objDetectDataMsg);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Draw classify result on video failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    } else if (outputType_ == "presentagent") {
        ret = SendImage(objDetectDataMsg);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Send image to presentAgent failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    } else if (outputType_ == "rtsp") {
        ret = SendImage(objDetectDataMsg);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Send image to rtsp failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    } else if (outputType_ == "stdout") {
        ret = PrintResult(objDetectDataMsg);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("stdout result on screen failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    } else if (outputType_ == "mqtt") {
        ret = SendImage(objDetectDataMsg);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Send image to mqtt failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    }
    else {
        ret = DrawResultOnPic(objDetectDataMsg);
        if (ret != ACLLITE_OK) {
            ACLLITE_LOG_ERROR("Send image to presentAgent failed, error %d", ret);
            return ACLLITE_ERROR;
        }
    }
    return ACLLITE_OK;
}

AclLiteError onlyDetectPostprocessThread::GetOutputFrameResolution(int &frameWidth, int &frameHeight, uint32_t channelId)
{
    std::string outputFrameWidthKey = "outputFrameWidth_" + to_string(channelId);
    std::string outputFrameHeightKey = "outputFrameHeight_" + to_string(channelId);

    std::map<std::string, std::string> config;
    if (!ReadConfig(config, configFile_)) {
        return ACLLITE_ERROR;
    }

    std::map<std::string, std::string>::const_iterator mIter = config.begin();
    for (; mIter != config.end(); ++mIter) {
        if (mIter->first == outputFrameWidthKey) {
            frameWidth = atoi(mIter->second.c_str());
            ACLLITE_LOG_INFO("video %d width %d",
                             channelId, frameWidth);
        } else if (mIter->first == outputFrameHeightKey) {
            frameHeight = atoi(mIter->second.c_str());
            ACLLITE_LOG_INFO("video %d height %d",
                             channelId, frameHeight);
        }
    }

    return ACLLITE_OK;
}

AclLiteError onlyDetectPostprocessThread::DisplayMsgSend(shared_ptr<ObjDetectDataMsg> objDetectDataMsg)
{
    AclLiteError ret;
    while (1) {
        if (outputType_ == "presentagent") {
#ifdef USE_PRESENT
            ret = SendMessage(objDetectDataMsg->presentAgentDisplayThreadId, MSG_PRESENT_AGENT_DISPLAY, objDetectDataMsg);
#endif
        }
        if (outputType_ == "rtsp") {
            ret = SendMessage(objDetectDataMsg->rtspDisplayThreadId, MSG_RTSP_DISPLAY, objDetectDataMsg);
        }
        if (outputType_ == "mqtt"){
            ret = SendMessage(objDetectDataMsg->mqttSendThreadId, MSG_MQTT_SEND, objDetectDataMsg);
        }
        if (ret == ACLLITE_ERROR_ENQUEUE) {
            usleep(500);
            continue;
        } else if (ret == ACLLITE_OK) {
            break;
        } else {
            ACLLITE_LOG_ERROR("Send present agent display message failed, error %d", ret);
            return ret;
        }
    }

    return ACLLITE_OK;
}

AclLiteError onlyDetectPostprocessThread::PrintResult(shared_ptr<ObjDetectDataMsg> &objDetectDataMsg)
{
    cout<<"[rtsp video "<<objDetectDataMsg->channelId<<"]: { ";
    for (int i = 0; i < objDetectDataMsg->objInfo.size(); ++i) {
        cout<<"[car"<<i<<",("<<objDetectDataMsg->objInfo[i].rectangle.lt.x
            <<","<<objDetectDataMsg->objInfo[i].rectangle.lt.y<<"),("
            <<objDetectDataMsg->objInfo[i].rectangle.rb.x
            <<","<<objDetectDataMsg->objInfo[i].rectangle.rb.y<<"),"
            <<objDetectDataMsg->objInfo[i].classify_result<<"] ";
    }
    cout<<"}"<<endl;
    return ACLLITE_OK;
}

AclLiteError onlyDetectPostprocessThread::SetOutputVideo()
{
    stringstream sstream;
    sstream.str("");
    sstream << "../out/output/out_test" << channelId_<<".mp4";
    int fps = 25;
    outputVideo_.open(sstream.str(), cv::VideoWriter::fourcc('m', 'p', '4', 'v'), fps, cv::Size(outputFrameWidth_, outputFrameHeight_));
    return ACLLITE_OK;
}

AclLiteError onlyDetectPostprocessThread::SendImage(shared_ptr<ObjDetectDataMsg> &objDetectDataMsg)
{
    for (int i = 0; i < objDetectDataMsg->objInfo.size(); ++i) {
        cv::rectangle(objDetectDataMsg->frame, objDetectDataMsg->objInfo[i].rectangle.lt,
                      objDetectDataMsg->objInfo[i].rectangle.rb,
                      kColors_[i % kColors_.size()], kLineSolid_);
        cv::putText(objDetectDataMsg->frame, objDetectDataMsg->objInfo[i].detect_result,
                    cv::Point(objDetectDataMsg->objInfo[i].rectangle.lt.x - kLabelOffset_,
                              objDetectDataMsg->objInfo[i].rectangle.lt.y - kLabelOffset_),
                    cv::FONT_HERSHEY_COMPLEX, kFountScale_, kFontColor_);
        cv::putText(objDetectDataMsg->frame, objDetectDataMsg->objInfo[i].classify_result,
                    cv::Point(objDetectDataMsg->objInfo[i].rectangle.lt.x,
                              objDetectDataMsg->objInfo[i].rectangle.lt.y + kLabelOffset_),
                    cv::FONT_HERSHEY_COMPLEX, kFountScale_, kFontColor_);
    }

    AclLiteError ret = DisplayMsgSend(objDetectDataMsg);
    if (ret != ACLLITE_OK) {
        ACLLITE_LOG_ERROR("Send display msg failed");
        return ACLLITE_ERROR;
    }

    return ACLLITE_OK;
}

AclLiteError onlyDetectPostprocessThread::DrawResultOnPic(shared_ptr<ObjDetectDataMsg> &objDetectDataMsg)
{

    for (int i = 0; i < objDetectDataMsg->objInfo.size(); ++i) {
        cv::rectangle(objDetectDataMsg->frame, objDetectDataMsg->objInfo[i].rectangle.lt,
                      objDetectDataMsg->objInfo[i].rectangle.rb,
                      kColors_[i % kColors_.size()], kLineSolid_);
        cv::putText(objDetectDataMsg->frame, objDetectDataMsg->objInfo[i].detect_result,
                    cv::Point(objDetectDataMsg->objInfo[i].rectangle.lt.x - kLabelOffset_,
                              objDetectDataMsg->objInfo[i].rectangle.lt.y - kLabelOffset_),
                    cv::FONT_HERSHEY_COMPLEX, kFountScale_, kFontColor_);
        cv::putText(objDetectDataMsg->frame, objDetectDataMsg->objInfo[i].classify_result,
                    cv::Point(objDetectDataMsg->objInfo[i].rectangle.lt.x,
                              objDetectDataMsg->objInfo[i].rectangle.lt.y + kLabelOffset_),
                    cv::FONT_HERSHEY_COMPLEX, kFountScale_, kFontColor_);
    }

    stringstream sstream;
    sstream.str("");
    sstream << "../out/output/device_" << objDetectDataMsg->channelId
            << "_out_pic_" << objDetectDataMsg->frameNum << ".jpg";
    cv::imwrite(sstream.str(), objDetectDataMsg->frame);
    return ACLLITE_OK;
}

AclLiteError onlyDetectPostprocessThread::DrawResultOnVideo(shared_ptr<ObjDetectDataMsg> &objDetectDataMsg)
{
    for (int i = 0; i < objDetectDataMsg->objInfo.size(); ++i) {
        cv::rectangle(objDetectDataMsg->frame, objDetectDataMsg->objInfo[i].rectangle.lt,
                      objDetectDataMsg->objInfo[i].rectangle.rb,
                      kColors_[i % kColors_.size()], kLineSolid_);
        cv::putText(objDetectDataMsg->frame, objDetectDataMsg->objInfo[i].detect_result,
                    cv::Point(objDetectDataMsg->objInfo[i].rectangle.lt.x - kLabelOffset_,
                              objDetectDataMsg->objInfo[i].rectangle.lt.y - kLabelOffset_),
                    cv::FONT_HERSHEY_COMPLEX, kFountScale_, kFontColor_);
        cv::putText(objDetectDataMsg->frame, objDetectDataMsg->objInfo[i].classify_result,
                    cv::Point(objDetectDataMsg->objInfo[i].rectangle.lt.x,
                              objDetectDataMsg->objInfo[i].rectangle.lt.y + kLabelOffset_),
                    cv::FONT_HERSHEY_COMPLEX, kFountScale_, kFontColor_);
    }
    resize(objDetectDataMsg->frame, objDetectDataMsg->frame,
           cv::Size(outputFrameWidth_, outputFrameHeight_),
           0, 0, cv::INTER_LINEAR);
    outputVideo_ << objDetectDataMsg->frame;
    return ACLLITE_OK;
}

onlyDetectPostprocessThread::onlyDetectPostprocessThread(const char*& configFile, int channelId,ModelInfo programinfo)
        :configFile_(configFile), channelId_(channelId), modelInfo_(programinfo)
{
}

