package com.censoft.censoftrongtong.controller;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.censoft.censoftrongtong.domain.LedgerRoom;
import com.censoft.censoftrongtong.domain.dto.LedgerRoomDto;
import com.censoft.censoftrongtong.service.ILedgerRoomService;
import com.censoft.censoftrongtong.service.IRiskPlanInherentListService;
import com.ruoyi.common.core.domain.R;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;

/**
 * 基础数据-楼层Controller
 *
 * @author ruoyi
 * @date 2023-06-26
 */
@RestController
@RequestMapping(value = {"/app-api/ledger/room", "/system/ledger/room"})
public class LedgerRoomController extends BaseController {
    @Autowired
    private ILedgerRoomService ledgerRoomService;

    @Resource
    private IRiskPlanInherentListService riskPlanInherentListService;

    /**
     * 查询基础数据-楼层列表
     */
    @GetMapping("/list")
    public AjaxResult list(LedgerRoom ledgerRoom) {
        ledgerRoom.setDelFlag("0");
        List<LedgerRoom> list = ledgerRoomService.selectLedgerRoomList(ledgerRoom);
        return success(list);
    }

    /**
     * 查询基础数据-楼层列表
     */
    @GetMapping("/listPc")
    public TableDataInfo listPc(LedgerRoomDto ledgerRoomDto) {
        startPage();
        if (ledgerRoomDto.getDeptId() == null){
            ledgerRoomDto.setDeptId(getDeptId());
        }
        List<LedgerRoomDto> list = ledgerRoomService.selectLedgerRoomDtoList(ledgerRoomDto);
        return getDataTable(list);
    }

    /**
     * 根据楼层id获取房间列表
     *
     * @param floorId 楼宇id
     * @real_return {@link R <List<LedgerRoom>>}
     */
    @GetMapping("/list/{floorId}")
    public R<List<LedgerRoom>> getRoomListByFloorId(@PathVariable("floorId") Long floorId) {
        return R.ok(ledgerRoomService.getRoomListByFloorId(floorId));
    }

    /**
     * 导出基础数据-楼层列表
     */
    @Log(title = "基础数据-楼层", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, LedgerRoomDto ledgerRoomDto) {
        if (ledgerRoomDto.getDeptId() == null){
            ledgerRoomDto.setDeptId(getDeptId());
        }
        List<LedgerRoomDto> list = ledgerRoomService.selectLedgerRoomDtoList(ledgerRoomDto);
        ExcelUtil<LedgerRoomDto> util = new ExcelUtil<LedgerRoomDto>(LedgerRoomDto.class);
        util.exportExcel(response, list, "基础数据-楼层数据");
    }

    /**
     * 获取基础数据-楼层详细信息
     */
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") Long id) {
        return success(ledgerRoomService.getById(id));
    }

    /**
     * 新增基础数据-楼层
     */
    @Log(title = "基础数据-楼层", businessType = BusinessType.INSERT)
    @PostMapping(value = "/save")
    public AjaxResult add(@RequestBody LedgerRoom ledgerRoom) {
        return toAjax(ledgerRoomService.save(ledgerRoom));
    }

    /**
     * 修改基础数据-楼层
     */
    @Log(title = "基础数据-楼层", businessType = BusinessType.UPDATE)
    @PostMapping("/update")
    public AjaxResult edit(@RequestBody LedgerRoom ledgerRoom) {
        return toAjax(ledgerRoomService.updateById(ledgerRoom));
    }

    /**
     * 批量修改基础数据-楼层
     */
    @Log(title = "基础数据-楼层", businessType = BusinessType.UPDATE)
    @PostMapping("/update/batch")
    public AjaxResult edit(@RequestBody List<LedgerRoom> ledgerRooms,Long floorId) {
        //1、先把id不在ledgerRooms中的设置为删除
        List<Long> ids = ledgerRooms.stream()
                .map(LedgerRoom::getId)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
        LambdaQueryWrapper<LedgerRoom> wrapper = new LambdaQueryWrapper();
        wrapper = wrapper.eq(LedgerRoom::getFloorId,floorId);
        if (!ids.isEmpty()){
            wrapper = wrapper.notIn(LedgerRoom::getId, ids);
        }
        ledgerRoomService.remove(wrapper);
        //删除固有现有风险数据
        //ledgerRoomService.deleteRoomByRoomIds(ids);

        //2、更新或插入ledgerRooms中的数据
        ledgerRooms.forEach(ledgerRoom -> {
            ledgerRoom.setFloorId(floorId);
        });
        ledgerRoomService.saveOrUpdateBatch(ledgerRooms);
        return toAjax(1);
    }

    /**
     * 删除基础数据-楼层
     */
    @Log(title = "基础数据-楼层", businessType = BusinessType.DELETE)
    @PostMapping("/delete/{ids}")
    public AjaxResult remove(@PathVariable Long[] ids) {
        QueryWrapper<LedgerRoom> queryWrapper = new QueryWrapper<>();
        queryWrapper.in("id", Arrays.asList(ids));
        ledgerRoomService.remove(queryWrapper);
        return AjaxResult.success();
    }

    /**
     * 删除基础数据-楼层
     */
    @Log(title = "基础数据-楼层", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public AjaxResult removeIds(@PathVariable Long[] ids)
    {
        ledgerRoomService.deleteRoomByRoomIds(Arrays.asList(ids));
        return AjaxResult.success();
    }
}
