package com.censoft.censoftrongtong.service.impl;

import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.censoft.censoftrongtong.domain.*;
import com.censoft.censoftrongtong.domain.dto.RiskExistingListSaveDto;
import com.censoft.censoftrongtong.domain.dto.RiskInherentListExportDto;
import com.censoft.censoftrongtong.domain.dto.RiskPlanAppExistingListDetailsDto;
import com.censoft.censoftrongtong.domain.dto.statistics.*;
import com.censoft.censoftrongtong.mapper.RiskPlanExistingListMapper;
import com.censoft.censoftrongtong.service.ILedgerBuildingService;
import com.censoft.censoftrongtong.service.ILedgerProjectService;
import com.censoft.censoftrongtong.service.IRiskPlanExistingListService;
import com.github.yulichang.base.MPJBaseServiceImpl;
import com.ruoyi.common.utils.DateUtils;
import com.censoft.censoftrongtong.domain.RiskLevelProportion;
import com.censoft.censoftrongtong.domain.RiskNumIndex;
import com.censoft.censoftrongtong.service.IRiskLevelProportionService;
import com.censoft.censoftrongtong.service.IRiskNumIndexService;
import com.ruoyi.system.service.ISysUploadFileService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 现有风险清单库Service业务层处理
 *
 * @author ruoyi
 * @date 2023-06-26
 */
@Service
public class RiskPlanExistingListServiceImpl
        extends MPJBaseServiceImpl<RiskPlanExistingListMapper, RiskPlanExistingList>
        implements IRiskPlanExistingListService {
    @Autowired
    private RiskPlanExistingListMapper riskPlanExistingListMapper;
    @Autowired
    private ILedgerBuildingService ledgerBuildingService;
    @Autowired
    private ILedgerProjectService projectService;
    @Autowired
    private IRiskLevelProportionService riskLevelProportionService;
    @Autowired
    private IRiskNumIndexService riskNumIndexService;
    @Resource
    private ISysUploadFileService uploadFileService;


    @Override
    public Long getNationalRiskNum(Integer level, String areaName, Date beginTime, Long peojectId) {

        return riskPlanExistingListMapper.getNationalRiskNum(level, areaName, beginTime, peojectId);
    }

    @Override
    public List<RiskIndexRankingDto> getRiskIndexRanking(Integer level, String areaName) {

        return riskPlanExistingListMapper.getRiskIndexRanking(level, areaName);
    }

    @Override
    public LinkedHashMap<String, Long> getAnnualRiskStatistics(Integer level, String areaName) {

        return riskPlanExistingListMapper.getAnnualRiskStatistics(level, areaName);
    }

    @Override
    public List<RiskLevelStatisticsDto> getRiskLevelStatistics(Integer level, String areaName, Long projectId) {
        return riskPlanExistingListMapper.getRiskLevelStatistics(level, areaName, projectId);
    }

    @Override
    public List<AccidentTypeStatisticsDto> getAccidentTypeStatistics(Integer level, String areaName) {
        return riskPlanExistingListMapper.getAccidentTypeStatistics(level, areaName);
    }

    @Override
    public List<RiskProjectRankingDto> getRiskProjectRanking(Integer level, String areaName) {
        return riskPlanExistingListMapper.getRiskProjectRanking(level, areaName);
    }

    @Override
    public List<ProjectRiskInfoDto> getProjectRiskInfo(Long projectId, Long buildingId, Long floorId, String level) {
        return riskPlanExistingListMapper.getProjectRiskInfo(projectId, buildingId, floorId, level);
    }

    @Override
    public List<RiskMapStatisticsDto> getRiskMapStatistics(MapStatisticsQueryDto queryDto) {
        return riskPlanExistingListMapper.getRiskMapStatistics(queryDto);
    }


    /**
     * 查询现有风险清单库
     *
     * @param id 现有风险清单库主键
     * @return 现有风险清单库
     */
    @Override
    public RiskPlanExistingList selectRiskPlanExistingListById(Long id) {
        return riskPlanExistingListMapper.selectRiskPlanExistingListById(id);
    }

    /**
     * 查询现有风险清单库列表
     *
     * @param riskPlanExistingList 现有风险清单库
     * @return 现有风险清单库
     */
    @Override
    public List<RiskPlanExistingList> selectRiskPlanExistingListList(RiskPlanExistingList riskPlanExistingList) {
        return riskPlanExistingListMapper.selectRiskPlanExistingListList(riskPlanExistingList);
    }


    /**
     * 新增现有风险清单库
     *
     * @param riskPlanExistingList 现有风险清单库
     * @return 结果
     */
    @Override
    public int insertRiskPlanExistingList(RiskPlanExistingList riskPlanExistingList) {
        riskPlanExistingList.setCreateTime(DateUtils.getNowDate());
        return riskPlanExistingListMapper.insertRiskPlanExistingList(riskPlanExistingList);
    }

    /**
     * 修改现有风险清单库
     *
     * @param riskPlanExistingList 现有风险清单库
     * @return 结果
     */
    @Override
    public int updateRiskPlanExistingList(RiskPlanExistingList riskPlanExistingList) {
        riskPlanExistingList.setUpdateTime(DateUtils.getNowDate());
        return riskPlanExistingListMapper.updateRiskPlanExistingList(riskPlanExistingList);
    }

    /**
     * 批量删除现有风险清单库
     *
     * @param ids 需要删除的现有风险清单库主键
     * @return 结果
     */
    @Override
    public int deleteRiskPlanExistingListByIds(Long[] ids) {
        return riskPlanExistingListMapper.deleteRiskPlanExistingListByIds(ids);
    }

    /**
     * 删除现有风险清单库信息
     *
     * @param id 现有风险清单库主键
     * @return 结果
     */
    @Override
    public int deleteRiskPlanExistingListById(Long id) {
        return riskPlanExistingListMapper.deleteRiskPlanExistingListById(id);
    }

    @Override
    public RiskPlanAppExistingListDetailsDto getRiskPlanAppExistingListDetailsDtoByExistingId(Long existingId) {
        RiskPlanAppExistingListDetailsDto dto = riskPlanExistingListMapper.getRiskPlanAppExistingListDetailsDtoByExistingId(existingId);

        //风险源照片
        if (!StrUtil.hasEmpty(dto.getPictureFileIds())) {
            List<Long> fileIds = Arrays.stream(dto.getPictureFileIds().split(","))
                    .map(Long::parseLong)
                    .collect(Collectors.toList());
            dto.setPictureFile(uploadFileService.selectSysFileByFileIds(fileIds));
        }
        //工程技术措施附件
        if (!StrUtil.hasEmpty(dto.getMeasuresProjectFileIds())) {
            List<Long> fileIds = Arrays.stream(dto.getMeasuresProjectFileIds().split(","))
                    .map(Long::parseLong)
                    .collect(Collectors.toList());
            dto.setMeasuresProjectFile(uploadFileService.selectSysFileByFileIds(fileIds));
        }
        //管理措施附件
        if (!StrUtil.hasEmpty(dto.getMeasuresAdministrationFileIds())) {
            List<Long> fileIds = Arrays.stream(dto.getMeasuresAdministrationFileIds().split(","))
                    .map(Long::parseLong)
                    .collect(Collectors.toList());
            dto.setMeasuresAdministrationFile(uploadFileService.selectSysFileByFileIds(fileIds));
        }
        //应急处置措施附件
        if (!StrUtil.hasEmpty(dto.getMeasuresEmergencyFileIds())) {
            List<Long> fileIds = Arrays.stream(dto.getMeasuresEmergencyFileIds().split(","))
                    .map(Long::parseLong)
                    .collect(Collectors.toList());
            dto.setMeasuresEmergencyFile(uploadFileService.selectSysFileByFileIds(fileIds));
        }
        return dto;
    }

    @Override
    public Boolean saveRiskExistingListSaveDto(RiskExistingListSaveDto saveDto) {
        RiskPlanExistingList existingList = new RiskPlanExistingList();
        BeanUtils.copyProperties(saveDto, existingList);
        existingList.setCreateTime(new Date());
        return saveOrUpdate(existingList);
    }


    /**
     * 计算房间风险等级
     * 计算规则 ：
     * 当前楼的所有现状风险
     * 0%-4%    重大
     * 4%-20%   较大
     * 20%-50%  一般
     * 50%-100% 低
     *
     * @param score
     * @return
     */
    @Override
    public Map<String, String> getRiskLevelByScore(Long buildingId, BigDecimal score) {
        Map<String, String> result = new HashMap<>();
        //查询当前楼栋现状风险列表
        RiskPlanExistingList query = new RiskPlanExistingList();
        query.setBuildingId(buildingId.toString());
        List<RiskPlanExistingList> existingList = this.selectRiskPlanExistingListList(query);
        if (existingList == null) {
            existingList = new ArrayList<>();
        }
        RiskPlanExistingList riskPlan = new RiskPlanExistingList();
        riskPlan.setLevelScore(score);
        riskPlan.setId(null);
        riskPlan.setBuildingId(buildingId.toString());
        existingList.add(riskPlan);
        existingList = existingList.stream().sorted(Comparator.comparing(RiskPlanExistingList::getLevelScore).reversed()).collect(Collectors.toList());
        //查询风险等级占比
        List<RiskLevelProportion> levelProportions = riskLevelProportionService.list();

        if (existingList.size() <= 0)
            return result;
        for (RiskLevelProportion item : levelProportions) {
            BigDecimal s = new BigDecimal(existingList.size());
            long min = s.multiply(item.getLevelScaleMin()).setScale(0, RoundingMode.UP).longValue();
            long max = s.multiply(item.getLevelScaleMax()).setScale(0, RoundingMode.UP).longValue();
            if (min == max && max == 1)
                continue;
            //风险列表排序后按照顺序设置风险等级
            List<RiskPlanExistingList> riskList = existingList.stream().skip(min).limit((max - min)).collect(Collectors.toList());
            for (RiskPlanExistingList risk : riskList) {
                if (risk.getId() == null) {
                    result.put("level", item.getRiskName());
                    result.put("levelScore", score.toString());
                    result.put("riskColor", item.getRiskColor());
                }
            }
        }
        return result;
    }


    /**
     * 计算房间风险等级
     * 计算规则 ：
     * 当前楼的所有现状风险
     * 0%-4%    重大
     * 4%-20%   较大
     * 20%-50%  一般
     * 50%-100% 低
     *
     * @param score
     * @return
     */
    @Override
    public Map<String, String> getRiskLevelByScoreAndUpd(Long buildingId, BigDecimal score) {
        Map<String, String> result = new HashMap<>();
        //查询当前楼栋现状风险列表
        RiskPlanExistingList query = new RiskPlanExistingList();
        query.setBuildingId(buildingId.toString());
        List<RiskPlanExistingList> existingList = this.selectRiskPlanExistingListList(query);
        if (existingList == null) {
            existingList = new ArrayList<>();
        }
        RiskPlanExistingList riskPlan = new RiskPlanExistingList();
        riskPlan.setLevelScore(score);
        riskPlan.setId(null);
        riskPlan.setBuildingId(buildingId.toString());
        existingList.add(riskPlan);
        existingList = existingList.stream().sorted(Comparator.comparing(RiskPlanExistingList::getLevelScore).reversed()).collect(Collectors.toList());
        //查询风险等级占比
        List<RiskLevelProportion> levelProportions = riskLevelProportionService.list();

        if (existingList.size() <= 0)
            return result;
        BigDecimal riskIndex = BigDecimal.ZERO;
        Map<String, List<RiskPlanExistingList>> map = new HashMap<>();
        for (RiskLevelProportion item : levelProportions) {
            BigDecimal s = new BigDecimal(existingList.size());
            long min = s.multiply(item.getLevelScaleMin()).setScale(0, RoundingMode.UP).longValue();
            long max = s.multiply(item.getLevelScaleMax()).setScale(0, RoundingMode.UP).longValue();
            if (min == max && max == 1)
                continue;
            //风险列表排序后按照顺序设置风险等级
            List<RiskPlanExistingList> riskList = existingList.stream().skip(min).limit((max - min)).collect(Collectors.toList());
            for (RiskPlanExistingList risk : riskList) {
                if (risk.getId() != null) {
                    risk.setLevel(item.getRiskName());
                    risk.setRiskColor(item.getRiskColor());
                    //更新风险等级
                    this.updateRiskPlanExistingList(risk);
                } else {
                    result.put("level", item.getRiskName());
                    result.put("levelScore", score.toString());
                    result.put("riskColor", item.getRiskColor());
                }
            }
            //计算每个类型的平均值及对应的风险指数
            BigDecimal riskAvg = riskList.stream().map(f -> f.getLevelScore() == null ? BigDecimal.ZERO : f.getLevelScore()).reduce(BigDecimal.ZERO, BigDecimal::add)
                    .divide(BigDecimal.valueOf(riskList.size()), 2, BigDecimal.ROUND_HALF_UP);
            //计算各等级风险指数
            BigDecimal thisRiskIndex = riskAvg.multiply(item.getLevelScaleMax()).multiply(item.getRiskGradeIndex());
            //求和计算总体风险指数
            riskIndex = riskIndex.add(thisRiskIndex);
        }
        LedgerBuilding building = ledgerBuildingService.getById(buildingId);

        //查询风险数量系数inherentList
        List<RiskNumIndex> riskNumIndexList = riskNumIndexService.list();
        for (RiskNumIndex numIndex : riskNumIndexList) {
            if (existingList.size() >= numIndex.getRiskNumIndexMin() && existingList.size() < numIndex.getRiskNumIndexMax()) {
                BigDecimal buildingIndex = numIndex.getRiskVal().multiply(riskIndex);
                building.setRiskIndex(buildingIndex);
                for (RiskLevelProportion item : levelProportions) {
                    if (buildingIndex.compareTo(item.getLevelValMin()) > -1 && buildingIndex.compareTo(item.getLevelValMax()) == -1) {
                        building.setRiskColor(item.getRiskColor());
                        building.setRiskLevel(item.getRiskName());
                    }
                }
                ledgerBuildingService.updateById(building);
            }
        }
        //计算楼栋的平均风险指数（=项目风险指数）
        List<LedgerBuilding> buildingList = ledgerBuildingService.list(Wrappers.<LedgerBuilding>query().lambda().eq(LedgerBuilding::getProjectId, building.getProjectId()));
        BigDecimal projectRiskIndex = buildingList.stream().map(f -> f.getRiskIndex() == null ? BigDecimal.ZERO : f.getRiskIndex()).reduce(BigDecimal.ZERO, BigDecimal::add)
                .divide(BigDecimal.valueOf(buildingList.size()), 2, BigDecimal.ROUND_HALF_UP);
        //计算当前项目的风险指数及等级
        LedgerProject project = projectService.getById(building.getProjectId());
        project.setRiskIndex(projectRiskIndex);
        for (RiskLevelProportion item : levelProportions) {
            if (projectRiskIndex.compareTo(item.getLevelValMin()) > -1 && projectRiskIndex.compareTo(item.getLevelValMax()) == -1) {
                project.setRiskColor(item.getRiskColor());
                project.setRiskLevel(item.getRiskName());
            }
        }
        projectService.updateById(project);
        return result;
    }


    @Override
    public List<RiskInherentListExportDto> getPlanExistingList(RiskInherentListExportDto riskInherentListExportDto) {
        return riskPlanExistingListMapper.getPlanExistingList(riskInherentListExportDto);
    }

}
