package com.censoft.censoftrongtong.controller;

import cn.hutool.core.convert.Convert;
import com.censoft.censoftrongtong.domain.LedgerBuilding;
import com.censoft.censoftrongtong.domain.RiskPlan;
import com.censoft.censoftrongtong.domain.dto.LedgerFloorDto;
import com.censoft.censoftrongtong.domain.dto.RiskInherentListExportDto;
import com.censoft.censoftrongtong.domain.dto.RiskNotificationExportWordDto;
import com.censoft.censoftrongtong.domain.dto.RiskPlanAppTaskDetailsDto;
import com.censoft.censoftrongtong.enums.RiskPlanStatusType;
import com.censoft.censoftrongtong.service.*;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.core.domain.R;
import com.ruoyi.common.core.domain.entity.SysUser;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.WordUtil;
import com.ruoyi.system.service.ISysUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 风险计划Controller
 *
 * @author ruoyi
 * @date 2023-06-26
 */
@RestController
@RequestMapping("/system/risk/plan")
public class RiskPlanController extends BaseController
{
    @Autowired
    private IRiskPlanService riskPlanService;

    @Resource
    private IRiskPlanInherentListService riskPlanInherentListService;

    @Autowired
    private IRiskPlanExistingListService riskPlanExistingListService;

    @Resource
    private ILedgerBuildingService ledgerBuildingService;

    @Resource
    private ISysUserService userService;

    @Resource
    private ILedgerFloorService ledgerFloorService;

    /**
     * 查询风险计划列表
     */
    @GetMapping("/list")
    public TableDataInfo list(RiskPlan riskPlan)
    {
        startPage();
        if(riskPlan.getCreateDeptId()==null) {
            if ("ledger".equals(riskPlan.getIsLedger())){
                riskPlan.setCreateDeptId(getDeptId());
            }else {
                riskPlan.setCreateUserId(getLoginUser().getUserId());
            }
        }

        List<RiskPlanAppTaskDetailsDto> list = riskPlanService.selectRiskPlanList(riskPlan);
        list.forEach(dto -> {
            dto.setStatusName(RiskPlanStatusType.getTitleByStatus(dto.getStatus()).getTitle());

            List<String> buildingIds = Arrays.asList(dto.getBuildingIds().split(","));
            String buildingNames = ledgerBuildingService.getBuildingListByIds(buildingIds)
                    .stream().map(LedgerBuilding::getName)
                    .collect(Collectors.joining(","));
            dto.setBuildingNames(buildingNames);

            List<String> workUserIds = Arrays.asList(dto.getWorkUserIds().split(","));
            String workUserNames = userService.getByIds(workUserIds)
                    .stream().map(SysUser::getNickName)
                    .collect(Collectors.joining(","));
            dto.setWorkUserNames(workUserNames);
        });


        return getDataTable(list);
    }

    /**
     * 导出风险计划列表
     */
    //@PreAuthorize("@ss.hasPermi('system:riskPlan:export')")
    //@Log(title = "风险计划", businessType = BusinessType.EXPORT)
    //@PostMapping("/export")
    //public void export(HttpServletResponse response, RiskPlan riskPlan)
    //{
    //    List<RiskPlan> list = riskPlanService.selectRiskPlanList(riskPlan);
    //    ExcelUtil<RiskPlan> util = new ExcelUtil<RiskPlan>(RiskPlan.class);
    //    util.exportExcel(response, list, "风险计划数据");
    //}

    /**
     * 获取风险计划详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:riskPlan:query')")
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") Long id)
    {
        return success(riskPlanService.selectRiskPlanById(id));
    }

    /**
     * 新增风险计划
     */
    @PreAuthorize("@ss.hasPermi('system:riskPlan:add')")
    @Log(title = "风险计划", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody RiskPlan riskPlan)
    {
        return toAjax(riskPlanService.insertRiskPlan(riskPlan));
    }

    /**
     * 修改风险计划
     */
    @PreAuthorize("@ss.hasPermi('system:riskPlan:edit')")
    @Log(title = "风险计划", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody RiskPlan riskPlan)
    {
        return toAjax(riskPlanService.updateRiskPlan(riskPlan));
    }

    /**
     * 删除风险计划
     */
    @PreAuthorize("@ss.hasPermi('system:riskPlan:remove')")
    @Log(title = "风险计划", businessType = BusinessType.DELETE)
	@DeleteMapping("/{ids}")
    public AjaxResult remove(@PathVariable Long[] ids)
    {
        return toAjax(riskPlanService.deleteRiskPlanByIds(ids));
    }





    @PostMapping("/exportWord/riskNotification/{inherentId}")
    public void exportWord(@PathVariable Long inherentId,
            HttpServletRequest request, HttpServletResponse response) throws IOException {

        RiskNotificationExportWordDto exportWordDto = riskPlanInherentListService.getRiskNotificationExportWordDto(inherentId);
        Map<String, Object> dataMap = Convert.convert(Map.class, exportWordDto);

        dataMap.put("title", "风险告知卡");
        dataMap.put("path", "D:/ruoyi/uploadPath/upload/");
        // 数据填装至模板，保存文件
        String name = WordUtil.createDoc(dataMap, "word.ftl");
        // word导出
        WordUtil.responseDownloadFile(request, response, "D:/ruoyi/uploadPath/upload/", name);
    }


    /**
     * 导出项目风险清单表
     */
    @PostMapping("/exportExcel/InherentList")
    public void export(RiskInherentListExportDto riskInherentListExportDto, HttpServletResponse response) {
        try {
            List<RiskInherentListExportDto> dtos = riskPlanInherentListService.getPlanInherentList(riskInherentListExportDto);
            for (RiskInherentListExportDto dto : dtos) {
                if ("1".equals(dto.getMajorHazardSource())){
                    dto.setMajorHazardSource("是");
                }else {
                    dto.setMajorHazardSource("否");
                    dto.setHazardSourceName("/");
                    dto.setMajorHazardDescription("/");
                }
            }
            riskPlanInherentListService.exportPlanInherentList(response,dtos);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 查询固有风险信息列表
     */
    @GetMapping("/inherentList")
    public TableDataInfo list(RiskInherentListExportDto riskInherentListExportDto)
    {
        startPage();
        List<RiskInherentListExportDto> dtos = riskPlanInherentListService.getPlanInherentList(riskInherentListExportDto);
        for (RiskInherentListExportDto dto : dtos) {
            if ("1".equals(dto.getMajorHazardSource())){
                dto.setMajorHazardSource("是");
            }else {
                dto.setMajorHazardSource("否");
                dto.setHazardSourceName("/");
                dto.setMajorHazardDescription("/");
            }
        }
        return getDataTable(dtos);
    }

    /**
     * 查询固有风险信息列表
     */
    @GetMapping("/existingList")
    public TableDataInfo getPlanExistingList(RiskInherentListExportDto riskInherentListExportDto)
    {
        startPage();
        List<RiskInherentListExportDto> dtos = riskPlanExistingListService.getPlanExistingList(riskInherentListExportDto);
        for (RiskInherentListExportDto dto : dtos) {
            if ("1".equals(dto.getMajorHazardSource())){
                dto.setMajorHazardSource("是");
            }else {
                dto.setMajorHazardSource("否");
                dto.setHazardSourceName("/");
                dto.setMajorHazardDescription("/");
            }
        }
        return getDataTable(dtos);
    }

    /**
     * 根据楼宇id获取楼层列表
     *
     * @param buildingId 楼宇id
     * @real_return {@link R <List< LedgerFloor >>}
     */
    @GetMapping("/floor/list/{buildingId}")
    public R<List<LedgerFloorDto>> getFloorListByBuildingId(@PathVariable("buildingId") Long buildingId) {
        return R.ok(ledgerFloorService.getFloorListByBuildingId(buildingId));
    }
}
