package com.censoft.censoftrongtong.controller;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.censoft.censoftrongtong.domain.*;
import com.censoft.censoftrongtong.domain.dto.LedgerFloorDto;
import com.censoft.censoftrongtong.domain.dto.statistics.*;
import com.censoft.censoftrongtong.service.*;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.R;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.utils.DateUtils;
import com.censoft.censoftrongtong.domain.RiskLevelProportion;
import com.censoft.censoftrongtong.service.IRiskLevelProportionService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;


/**
 * 大屏接口
 */
@RestController
@RequestMapping("/system/bigScreen")
public class BigScreenController extends BaseController {

    @Autowired
    private IRiskLevelProportionService riskLevelProportionService;
    @Autowired
    ILedgerProjectService ledgerProjectService;
    @Autowired
    ILedgerBuildingService ledgerBuildingService;
    @Autowired
    private ILedgerFloorService ledgerFloorService;
    @Autowired
    private ILedgerRoomService ledgerRoomService;
    @Autowired
    private IRiskPlanExistingListService riskPlanExistingListService;
    @Autowired
    private IRiskPlanInherentListService riskPlanInherentListService;
    @Autowired
    private IRiskPlanService riskPlanServices;

    /**
     * 项目信息列表
     */
    @GetMapping("/selectLedgerProjectList")
    public R<TableDataInfo> selectLedgerProjectList(String name) {
        startPage();
        List<LedgerProject> list = ledgerProjectService.getLedgerProjectList(name, null, null);
        return R.ok(getDataTable(list));
    }

    /**
     * 项目，楼宇，楼层统计
     */
    @GetMapping("/evaluationDataStatistics")
    public R<EvaluationDataDto> evaluationDataStatistics(Integer level, String areaName) {
        if (areaName != null && areaName.contains("北京") || areaName.contains("重庆") || areaName.contains("上海") || areaName.contains("天津")) {
            level += 1;
        }
        EvaluationDataDto result = new EvaluationDataDto();
        Long projectNum = ledgerProjectService.getProjectCount(level, areaName);
        Long buildingNum = ledgerProjectService.getBuildingCount(level, areaName, null);
        Long floorNum = ledgerProjectService.getFloorCount(level, areaName, null);
        result.setProjectNum(projectNum);
        result.setBuildingNum(buildingNum);
        result.setFloorNum(floorNum);
        return R.ok(result);
    }

    /**
     * 全国风险，年度新增
     */
    @GetMapping("/nationalRiskStatistics")
    public R<NationalRiskStatisticsDto> nationalRiskStatistics(Integer level, String areaName, Integer type) {
        if (areaName != null && areaName.contains("北京") || areaName.contains("重庆") || areaName.contains("上海") || areaName.contains("天津")) {
            level += 1;
        }
        NationalRiskStatisticsDto result = new NationalRiskStatisticsDto();
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.MONTH, Calendar.JANUARY);
        calendar.set(Calendar.DAY_OF_MONTH, 1);
        Date beginTime = calendar.getTime();
        Long nationalRiskNum = riskPlanExistingListService.getNationalRiskNum(level, areaName, null, null);
        Long newRiskNum = riskPlanExistingListService.getNationalRiskNum(level, areaName, beginTime, null);
        result.setNationalRiskNum(nationalRiskNum);
        result.setNewRiskNum(newRiskNum);
        return R.ok(result);
    }

    /**
     * 总体风险指数排名
     */
    @GetMapping("/riskIndexRanking")
    public R<List<RiskIndexRankingDto>> riskIndexRanking(Integer level, String areaName, Integer type) {
        if (areaName != null && areaName.contains("北京") || areaName.contains("重庆") || areaName.contains("上海") || areaName.contains("天津")) {
            level += 1;
        }
        startPage();
        List<RiskIndexRankingDto> result = riskPlanExistingListService.getRiskIndexRanking(level, areaName);
        return R.ok(result);
    }

    /**
     * 年度风险数量统计
     */
    @GetMapping("/annualRiskStatistics")
    public R<LinkedHashMap<String, Long>> annualRiskStatistics(Integer level, String areaName, Integer type) {
        if (areaName != null && areaName.contains("北京") || areaName.contains("重庆") || areaName.contains("上海") || areaName.contains("天津")) {
            level += 1;
        }
        LinkedHashMap<String, Long> result = riskPlanExistingListService.getAnnualRiskStatistics(level, areaName);
        return R.ok(result);
    }

    /**
     * 风险级别统计
     */
    @GetMapping("/riskLevelStatistics")
    public R<List<RiskLevelStatisticsDto>> riskLevelStatistics(Integer level, String areaName, Integer type) {
        if (areaName != null && areaName.contains("北京") || areaName.contains("重庆") || areaName.contains("上海") || areaName.contains("天津")) {
            level += 1;
        }
        List<RiskLevelStatisticsDto> result = riskPlanExistingListService.getRiskLevelStatistics(level, areaName, null);
        return R.ok(result);
    }

    /**
     * 事故类型统计
     */
    @GetMapping("/accidentTypeStatistics")
    public R<List<AccidentTypeStatisticsDto>> accidentTypeStatistics(Integer level, String areaName, Integer type) {
        if (areaName != null && areaName.contains("北京") || areaName.contains("重庆") || areaName.contains("上海") || areaName.contains("天津")) {
            level += 1;
        }
        List<AccidentTypeStatisticsDto> result = riskPlanExistingListService.getAccidentTypeStatistics(level, areaName);
        return R.ok(result);
    }

    /**
     * 项目风险级别排名
     *
     * @param areaName
     * @param level
     * @param type
     * @return
     */
    @GetMapping("/riskProjectRanking")
    public R<List<RiskProjectRankingDto>> riskProjectRanking(Integer level, String areaName, Integer type) {
        if (areaName != null && areaName.contains("北京") || areaName.contains("重庆") || areaName.contains("上海") || areaName.contains("天津")) {
            level += 1;
        }
        List<RiskProjectRankingDto> result = riskPlanExistingListService.getRiskProjectRanking(level, areaName);
        return R.ok(result);
    }

    /**
     * 获取地图统计数据
     *
     * @param queryDto
     * @return
     */
    @GetMapping("/getRiskMapStatistics")
    public R<List<RiskMapStatisticsDto>> getRiskMapStatistics(MapStatisticsQueryDto queryDto) {
        if (queryDto.getAreaName() != null && queryDto.getAreaName().contains("北京") || queryDto.getAreaName().contains("重庆") || queryDto.getAreaName().contains("上海") || queryDto.getAreaName().contains("天津")) {
            queryDto.setLevel(queryDto.getLevel() + 1);
        }
        //查询风险等级占比
        List<RiskLevelProportion> levelProportions = riskLevelProportionService.list();
        List<RiskMapStatisticsDto> list = riskPlanExistingListService.getRiskMapStatistics(queryDto);
        List<MapLedgerProjectDto> projects = ledgerProjectService.getProjectListAndCount(queryDto.getLevel(), queryDto.getAreaName());
        for (RiskMapStatisticsDto r : list) {
            if (r.getAreaName() != null) {
                List<MapLedgerProjectDto> projectList = projects.stream().filter(s ->
                        (queryDto.getLevel().equals(0) && s.getProvince() != null && s.getProvince().contains(r.getAreaName()))
                                || (queryDto.getLevel().equals(1) && s.getCity() != null && s.getCity().contains(r.getAreaName()))
                                || (queryDto.getLevel().equals(2) && s.getDistrict() != null && s.getDistrict().contains(r.getAreaName()))).collect(Collectors.toList());


                if (projectList != null && projectList.size() > 0) {
                    BigDecimal projectRiskIndex = projectList.stream().map(f -> f.getRiskIndex() == null ? BigDecimal.ZERO : f.getRiskIndex()).reduce(BigDecimal.ZERO, BigDecimal::add)
                            .divide(BigDecimal.valueOf(projectList.size()), 2, BigDecimal.ROUND_HALF_UP);

                    for (RiskLevelProportion item : levelProportions) {
                        if (projectRiskIndex.compareTo(item.getLevelValMin()) > -1 && projectRiskIndex.compareTo(item.getLevelValMax()) == -1) {
                            r.setAreaColor(item.getRiskColor());
                        }
                    }
                    if (queryDto.getLevel().equals(2))
                        r.setProjectList(projectList);
                }
                else{

                }
            }
        }
        return R.ok(list);
    }


    /**
     * 项目级-上方横条(评估时间、评估数据、风险总数量、风险等级统计（现状、固有）)
     */
    @GetMapping("/projectStatisticsDetail")
    public R<ProjectStatisticsDetailDto> projectStatisticsDetail(@RequestParam Long projectId) {
        ProjectStatisticsDetailDto result = new ProjectStatisticsDetailDto();
        //项目名称
        LedgerProject project = ledgerProjectService.getById(projectId);
        result.setProjectName(project.getName());
        //评估时间
        List<RiskPlan> lastRiskPlan = riskPlanServices.list(Wrappers.<RiskPlan>query().lambda().eq(RiskPlan::getProjectId, projectId).orderByDesc(RiskPlan::getCreateTime));
        if (lastRiskPlan.size()>0) {
            result.setEvaluationTime(DateUtils.parseDateToStr(DateUtils.YYYY_MM_DD_HH_MM_SS, lastRiskPlan.get(0).getCreateTime()));
        }
        //评估数据
        Long buildingNum = ledgerProjectService.getBuildingCount(null, null, projectId);
        Long floorNum = ledgerProjectService.getFloorCount(null, null, projectId);
        EvaluationDataDto evaluationDataDto = new EvaluationDataDto();
        evaluationDataDto.setProjectNum(1L);
        evaluationDataDto.setBuildingNum(buildingNum);
        evaluationDataDto.setFloorNum(floorNum);
        result.setEvaluationData(evaluationDataDto);
        //固有风险数量
        Long inherentNum = riskPlanInherentListService.getNationalRiskNum(null, null, null, projectId);
        result.setInherentNum(inherentNum);
        //现状风险数量
        Long existingNum = riskPlanExistingListService.getNationalRiskNum(null, null, null, projectId);
        result.setExistingNum(existingNum);
        //风险总数量
        Long sumNum = inherentNum + existingNum;
        result.setRiskSumNum(sumNum);
        //风险等级-固有
        List<RiskLevelStatisticsDto> inherentLevelStatics = riskPlanInherentListService.getRiskLevelStatistics(null, null, projectId);
        result.setInherentLevelStatics(inherentLevelStatics);
        //风险等级-现状
        List<RiskLevelStatisticsDto> existingLevelStatics = riskPlanExistingListService.getRiskLevelStatistics(null, null, projectId);
        result.setExistingLevelStatics(existingLevelStatics);
        return R.ok(result);
    }

    /**
     * 项目风险信息
     */
    @GetMapping("/projectRiskInformation")
    public R<List<ProjectRiskInfoDto>> projectRiskInfo(Long projectId, Long buildingId, Long floorId, Integer type, String level) {
        List<ProjectRiskInfoDto> list = riskPlanInherentListService.getProjectRiskInfo(projectId, buildingId, floorId, level);
        return R.ok(list);
    }

    /**
     * 获取项目下的楼宇
     *
     * @param projectId 项目id
     * @param type      类型 地上、地下
     * @real_return {@link R<List<LedgerBuilding>>}
     */
    @GetMapping("/building/list/{projectId}")
    public R<List<LedgerBuilding>> getBuildingListByProjectId(@PathVariable("projectId") Long projectId, String type) {
        return R.ok(ledgerBuildingService.getBuildingListByProjectId(projectId, type));
    }


    /**
     * 根据楼宇id获取楼层列表
     *
     * @param buildingId 楼宇id
     * @real_return {@link R<List<LedgerFloor>>}
     */
    @GetMapping("/floor/list/{buildingId}")
    public R<List<LedgerFloorDto>> getFloorListByBuildingId(@PathVariable("buildingId") Long buildingId) {
        return R.ok(ledgerFloorService.getFloorListByBuildingId(buildingId));
    }

    /**
     * 根据楼层id获取房间列表
     *
     * @param floorId 楼宇id
     * @real_return {@link R<List<LedgerRoom>>}
     */
    @GetMapping("/room/list/{floorId}")
    public R<List<LedgerRoom>> getRoomListByFloorId(@PathVariable("floorId") Long floorId) {
        return R.ok(ledgerRoomService.getRoomListByFloorId(floorId));
    }


}
