package com.censoft.censoftrongtong.controller;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.convert.Convert;
import com.censoft.censoftrongtong.domain.LedgerBuilding;
import com.censoft.censoftrongtong.domain.RiskPlan;
import com.censoft.censoftrongtong.domain.dto.LedgerFloorDto;
import com.censoft.censoftrongtong.domain.dto.RiskInherentListExportDto;
import com.censoft.censoftrongtong.domain.dto.RiskNotificationExportWordDto;
import com.censoft.censoftrongtong.domain.dto.RiskPlanAppTaskDetailsDto;
import com.censoft.censoftrongtong.domain.vo.RiskPlanExportVO;
import com.censoft.censoftrongtong.domain.vo.RiskTypeExportVO;
import com.censoft.censoftrongtong.enums.RiskPlanStatusType;
import com.censoft.censoftrongtong.service.*;
import com.deepoove.poi.XWPFTemplate;
import com.deepoove.poi.config.Configure;
import com.deepoove.poi.config.ConfigureBuilder;
import com.deepoove.poi.data.ChartMultiSeriesRenderData;
import com.deepoove.poi.data.Charts;
import com.deepoove.poi.plugin.table.LoopRowTableRenderPolicy;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.core.domain.R;
import com.ruoyi.common.core.domain.entity.SysUser;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.WordUtil;
import com.ruoyi.system.service.ISysUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 风险计划Controller
 *
 * @author ruoyi
 * @date 2023-06-26
 */
@RestController
@RequestMapping({"/system/risk/plan","app-api/system/risk/plan"})
public class RiskPlanController extends BaseController
{
    @Autowired
    private IRiskPlanService riskPlanService;

    @Resource
    private IRiskPlanInherentListService riskPlanInherentListService;

    @Autowired
    private IRiskPlanExistingListService riskPlanExistingListService;

    @Resource
    private ILedgerBuildingService ledgerBuildingService;

    @Resource
    private ISysUserService userService;

    @Resource
    private ILedgerFloorService ledgerFloorService;

    /**
     * 查询风险计划列表
     */
    @GetMapping("/list")
    public TableDataInfo list(RiskPlan riskPlan)
    {
        startPage();
        if(riskPlan.getCreateDeptId()==null) {
            if ("ledger".equals(riskPlan.getIsLedger())){
                riskPlan.setCreateDeptId(getDeptId());
            }else {
                riskPlan.setCreateUserId(getLoginUser().getUserId());
            }
        }

        List<RiskPlanAppTaskDetailsDto> list = riskPlanService.selectRiskPlanList(riskPlan);
        list.forEach(dto -> {
            dto.setStatusName(RiskPlanStatusType.getTitleByStatus(dto.getStatus()).getTitle());

            List<String> buildingIds = Arrays.asList(dto.getBuildingIds().split(","));
            String buildingNames = ledgerBuildingService.getBuildingListByIds(buildingIds)
                    .stream().map(LedgerBuilding::getName)
                    .collect(Collectors.joining(","));
            dto.setBuildingNames(buildingNames);

            List<String> workUserIds = Arrays.asList(dto.getWorkUserIds().split(","));
            String workUserNames = userService.getByIds(workUserIds)
                    .stream().map(SysUser::getNickName)
                    .collect(Collectors.joining(","));
            dto.setWorkUserNames(workUserNames);
        });


        return getDataTable(list);
    }

    /**
     * 导出风险计划列表
     */
    //@PreAuthorize("@ss.hasPermi('system:riskPlan:export')")
    //@Log(title = "风险计划", businessType = BusinessType.EXPORT)
    //@PostMapping("/export")
    //public void export(HttpServletResponse response, RiskPlan riskPlan)
    //{
    //    List<RiskPlan> list = riskPlanService.selectRiskPlanList(riskPlan);
    //    ExcelUtil<RiskPlan> util = new ExcelUtil<RiskPlan>(RiskPlan.class);
    //    util.exportExcel(response, list, "风险计划数据");
    //}

    /**
     * 获取风险计划详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:riskPlan:query')")
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") Long id)
    {
        return success(riskPlanService.selectRiskPlanById(id));
    }

    /**
     * 新增风险计划
     */
    @PreAuthorize("@ss.hasPermi('system:riskPlan:add')")
    @Log(title = "风险计划", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody RiskPlan riskPlan)
    {
        return toAjax(riskPlanService.insertRiskPlan(riskPlan));
    }

    /**
     * 修改风险计划
     */
    @PreAuthorize("@ss.hasPermi('system:riskPlan:edit')")
    @Log(title = "风险计划", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody RiskPlan riskPlan)
    {
        return toAjax(riskPlanService.updateRiskPlan(riskPlan));
    }

    /**
     * 删除风险计划
     */
    @PreAuthorize("@ss.hasPermi('system:riskPlan:remove')")
    @Log(title = "风险计划", businessType = BusinessType.DELETE)
	@DeleteMapping("/{ids}")
    public AjaxResult remove(@PathVariable Long[] ids)
    {
        return toAjax(riskPlanService.deleteRiskPlanByIds(ids));
    }





    @PostMapping("/exportWord/riskNotification/{inherentId}")
    public void exportWord(@PathVariable Long inherentId,
            HttpServletRequest request, HttpServletResponse response) throws IOException {

        RiskNotificationExportWordDto exportWordDto = riskPlanInherentListService.getRiskNotificationExportWordDto(inherentId);
        Map<String, Object> dataMap = Convert.convert(Map.class, exportWordDto);

        dataMap.put("title", "风险告知卡");
        dataMap.put("path", "D:/ruoyi/uploadPath/upload/");
        // 数据填装至模板，保存文件
        String name = WordUtil.createDoc(dataMap, "word.ftl");
        // word导出
        WordUtil.responseDownloadFile(request, response, "D:/ruoyi/uploadPath/upload/", name);
    }


    /**
     * 导出项目风险清单表
     */
    @PostMapping("/exportExcel/InherentList")
    public void export(RiskInherentListExportDto riskInherentListExportDto, HttpServletResponse response) {
        try {
            List<RiskInherentListExportDto> dtos = riskPlanInherentListService.getPlanInherentList(riskInherentListExportDto);
            for (RiskInherentListExportDto dto : dtos) {
                if ("1".equals(dto.getMajorHazardSource())){
                    dto.setMajorHazardSource("是");
                }else {
                    dto.setMajorHazardSource("否");
                    dto.setHazardSourceName("/");
                    dto.setMajorHazardDescription("/");
                }
            }
            riskPlanInherentListService.exportPlanInherentList(response,dtos);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 查询固有风险信息列表
     */
    @GetMapping("/inherentList")
    public TableDataInfo list(RiskInherentListExportDto riskInherentListExportDto)
    {
        startPage();
        List<RiskInherentListExportDto> dtos = riskPlanInherentListService.getPlanInherentList(riskInherentListExportDto);
        for (RiskInherentListExportDto dto : dtos) {
            if ("1".equals(dto.getMajorHazardSource())){
                dto.setMajorHazardSource("是");
            }else {
                dto.setMajorHazardSource("否");
                dto.setHazardSourceName("/");
                dto.setMajorHazardDescription("/");
            }
        }
        return getDataTable(dtos);
    }

    /**
     * 查询固有风险信息列表
     */
    @GetMapping("/existingList")
    public TableDataInfo getPlanExistingList(RiskInherentListExportDto riskInherentListExportDto)
    {
        startPage();
        List<RiskInherentListExportDto> dtos = riskPlanExistingListService.getPlanExistingList(riskInherentListExportDto);
        for (RiskInherentListExportDto dto : dtos) {
            if ("1".equals(dto.getMajorHazardSource())){
                dto.setMajorHazardSource("是");
            }else {
                dto.setMajorHazardSource("否");
                dto.setHazardSourceName("/");
                dto.setMajorHazardDescription("/");
            }
        }
        return getDataTable(dtos);
    }

    /**
     * 根据楼宇id获取楼层列表
     *
     * @param buildingId 楼宇id
     * @real_return {@link R <List< LedgerFloor >>}
     */
    @GetMapping("/floor/list/{buildingId}")
    public R<List<LedgerFloorDto>> getFloorListByBuildingId(@PathVariable("buildingId") Long buildingId) {
        return R.ok(ledgerFloorService.getFloorListByBuildingId(buildingId));
    }

    @Log(title = "现状风险巡查用户请假", businessType = BusinessType.INSERT)
    @PostMapping("/advise")
    public AjaxResult advise(@RequestBody RiskPlan reqVO) {
        RiskPlan riskPlan = riskPlanService.selectRiskPlanById(reqVO.getId());
        riskPlan.setRemark(reqVO.getValue());
        riskPlanService.updateRiskPlan(riskPlan);
        return AjaxResult.success();
    }

    @GetMapping("/export/{id}")
    public void export(@PathVariable("id") Long id,HttpServletResponse response) {
        //模板地址，存放在resources目录下
        String filePath = "templates/word/风险评估报告模板.docx";
        //使用poi-tl进行模板处理
        ConfigureBuilder builder = Configure.builder();
        builder.useSpringEL(true);
        //执行循环策略
        LoopRowTableRenderPolicy strategy = new LoopRowTableRenderPolicy();
        //绑定集合对象
        //builder.bind("riskTypeList", strategy);
        builder.bind("table", strategy);
        //获取模板文件流
        InputStream inputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(filePath);
        assert inputStream != null;
        //组装数据
        RiskPlanExportVO data = riskPlanService.riskPlanExport(id);
        Map<String,Object> map = BeanUtil.beanToMap(data);
        //第一个图标: 风险点分级统计数量图
        ChartMultiSeriesRenderData chart = Charts
                .ofMultiSeries("风险点分级统计数量图", new String[] { "重大风险", "较大风险", "一般风险", "低风险" })
                .addSeries("riskType", new Integer[] { Integer.parseInt(data.getZdCount()), Integer.parseInt(data.getJdCount()),
                        Integer.parseInt(data.getYbCount()),Integer.parseInt(data.getDCount())})
                .create();
        map.put("riskType",chart);
        //第二个图表：不同事故类型对应的风险点数量统计图
        String[] riskPointName = new String[]{};
        Integer[] riskPointNum = new Integer[]{};
        if(!CollectionUtils.isEmpty(data.getRiskTypeList())){
            int size = data.getRiskTypeList().size();
            riskPointName = new String[size];
            riskPointNum = new Integer[size];
            List<RiskTypeExportVO> riskTypeList = data.getRiskTypeList().stream()
                    .sorted(Comparator.comparing(RiskTypeExportVO::getRiskTypeNum).reversed())
                    .collect(Collectors.toList());
            //占多数的风险点数量
            int maxRiskPointNum = 0;
            //风险点类型总数
            int totalRiskPointNum = 0;
            //占多数风险点名称
            String maxRiskPointName = "";
            //占对数比例
            String maxRiskRate = "";
            for (int i = 0; i < size; i++) {
                RiskTypeExportVO riskType = riskTypeList.get(i);
                riskPointNum[i] = riskType.getRiskTypeNum();
                riskPointName[i] = riskType.getRiskTypeName();
                totalRiskPointNum += riskType.getRiskTypeNum();
            }
            if(size >3){
                maxRiskPointNum = riskTypeList.get(0).getRiskTypeNum() + riskTypeList.get(1).getRiskTypeNum()
                                +riskTypeList.get(2).getRiskTypeNum();
                maxRiskPointName = riskTypeList.get(0).getRiskTypeName() + ","+ riskTypeList.get(1).getRiskTypeName()+","
                        +riskTypeList.get(2).getRiskTypeName();
            }else{
                maxRiskPointNum = riskTypeList.get(0).getRiskTypeNum();
                maxRiskPointName = riskTypeList.get(0).getRiskTypeName();
            }
            maxRiskRate = new BigDecimal(maxRiskPointNum).divide(new BigDecimal(totalRiskPointNum))
                    .multiply(new BigDecimal("100")).setScale(2, RoundingMode.HALF_UP).toString();
            map.put("maxRiskRate",maxRiskRate+"%");
            map.put("totalRiskPointNum",totalRiskPointNum);
            map.put("maxRiskPointName",maxRiskPointName);
        };
        ChartMultiSeriesRenderData chart2 = Charts
                .ofMultiSeries("不同事故类型对应的风险点数量统计图",riskPointName)
                .addSeries("riskPoint", riskPointNum)
                .create();
        map.put("riskPoint",chart2);
        XWPFTemplate render = XWPFTemplate.compile(inputStream, builder.build()).render(map);
        // 设置强制下载不打开org.apache.poi.xwpf.usermodel.XWPFDocument
        response.setContentType("application/force-download");
        response.addHeader("Access-Control-Expose-Headers", " Content-Disposition");
        //导出为word
        response.addHeader("Content-Disposition", "attachment; fileName=xsss.docx");
        try {
            render.write(response.getOutputStream());
        } catch (IOException e) {
            e.printStackTrace();
        }
        //导出为pdf
//        response.addHeader("Content-Disposition", "attachment; fileName=" + new String(("导出模板.pdf").getBytes(StandardCharsets.UTF_8), StandardCharsets.ISO_8859_1));
//        BufferedOutputStream outputStream = new BufferedOutputStream(response.getOutputStream());
//        //设置临时文件的地址
//        String tempPath = UUID.randomUUID() + ".docx";
//        //根据模板生成临时文件
//        render.writeToFile(tempPath);
//        //将docx流转换为pdf流
//        FileInputStream fileInputStream = new FileInputStream(tempPath);
//        WordConvertPdf.getPdfStreamByWordStream(fileInputStream, outputStream);
//        outputStream.flush();
//        outputStream.close();
//        fileInputStream.close();
//        //删除临时文件
//        File tempFile = new File(tempPath);
//        Files.delete(tempFile.toPath());
//        log.debug("删除临时word文件：{}", tempPath);
    }
}
