package com.censoft.censoftrongtong.controller;

import com.censoft.censoftrongtong.service.IBacklogInfoService;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.core.page.TableDataInfo;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.common.utils.StringUtils;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.system.domain.BacklogInfo;
import com.ruoyi.system.service.IMessageInfoService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 待办信息Controller
 *
 * @author ruoyi
 * @date 2024-01-09
 */
@RestController
@RequestMapping(value = {"/system/backlogInfo","/app-api/backlogInfo"})
public class BacklogInfoController extends BaseController {

    @Autowired
    private IBacklogInfoService backlogInfoService;

    @Autowired
    private IMessageInfoService messageInfoService;

    /**
     * 查询待办信息列表
     */
    @PreAuthorize("@ss.hasPermi('system:backlogInfo:list')")
    @PostMapping("/list")
    public TableDataInfo list(BacklogInfo backlogInfo) {
        startPage();
        String loginName = getUsername();
        if (StringUtils.isNotNull(backlogInfo.getParams().get("beginTime")) && !"".equals(backlogInfo.getParams().get("beginTime"))) {
            String beginTime = backlogInfo.getParams().get("beginTime") + " 00:00:00";
            backlogInfo.setBeginTime(beginTime);
        }
        if (StringUtils.isNotNull(backlogInfo.getParams().get("endTime")) && !"".equals(backlogInfo.getParams().get("endTime"))) {
            String endTime = backlogInfo.getParams().get("endTime") + " 59:59:59";
            backlogInfo.setEndTime(endTime);
        }
        backlogInfo.setBacklogBy(loginName);
        backlogInfo.setMenuUrl("menu_url");
        backlogInfo.setAppUrl("app_url");
        List<BacklogInfo> list = backlogInfoService.selectBacklogInfoList(backlogInfo);
        for (BacklogInfo info : list) {
            String createBy = info.getCreateBy() == null ? "系统" : info.getCreateBy();
            info.setCreateBy(createBy);
        }
        return getDataTable(list);
    }


    /**
     * 查询未读信息数量
     */
    @GetMapping("/unReadCount")
    @ResponseBody
    public AjaxResult unReadCount() {
        Map<String, Integer> map = new HashMap<>();
        String loginName = getUsername();
        int backlogInfo = backlogInfoService.selectUnReadCount(loginName, "app_url", "menu_url");
        int messageInfo = messageInfoService.selectUnReadCount(loginName);
        map.put("待办事项", backlogInfo);
        map.put("消息信息", messageInfo);
        return AjaxResult.success(map);
    }

    /**
     * 清除未读
     */
    @GetMapping("/changeStatus")
    @ResponseBody
    public AjaxResult changeStatus() {
        String loginName = getUsername();
        return toAjax(backlogInfoService.updateChangeStatus(loginName));
    }

    /**
     * 查询待办信息各状态数量
     */
    @GetMapping("/status/count")
    @ResponseBody
    public AjaxResult getStatusCount(String beginTime,String endTime,String keyword) {
        String loginName = getUsername();
        if (StringUtils.isNotEmpty(beginTime)) {
            beginTime = beginTime + " 00:00:00";
        }
        if (StringUtils.isNotEmpty(endTime)) {
            endTime = endTime + " 59:59:59";
        }
        return AjaxResult.success(backlogInfoService.getStatusCount(loginName,beginTime,endTime,keyword));
    }


    /**
     * 导出待办信息列表
     */
    @PreAuthorize("@ss.hasPermi('system:backlogInfo:export')")
    @Log(title = "待办信息", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, BacklogInfo backlogInfo) {
        List<BacklogInfo> list = backlogInfoService.selectBacklogInfoList(backlogInfo);
        ExcelUtil<BacklogInfo> util = new ExcelUtil<BacklogInfo>(BacklogInfo.class);
        util.exportExcel(response, list, "待办信息数据");
    }

    /**
     * 获取待办信息详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:backlogInfo:query')")
    @GetMapping(value = "/{id}")
    public AjaxResult getInfo(@PathVariable("id") String id) {
        return success(backlogInfoService.selectBacklogInfoById(id));
    }

    /**
     * 新增待办信息
     */
    @PreAuthorize("@ss.hasPermi('system:backlogInfo:add')")
    @Log(title = "待办信息", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody BacklogInfo backlogInfo) {
        return toAjax(backlogInfoService.insertBacklogInfo(backlogInfo));
    }

    /**
     * 修改待办信息
     */
    @PreAuthorize("@ss.hasPermi('system:backlogInfo:edit')")
    @Log(title = "待办信息", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody BacklogInfo backlogInfo) {
        return toAjax(backlogInfoService.updateBacklogInfo(backlogInfo));
    }

    /**
     * 删除待办信息
     */
    @PreAuthorize("@ss.hasPermi('system:backlogInfo:remove')")
    @Log(title = "待办信息", businessType = BusinessType.DELETE)
    @DeleteMapping("/{ids}")
    public AjaxResult remove(@PathVariable String[] ids) {
        return toAjax(backlogInfoService.deleteBacklogInfoByIds(ids));
    }
}
