package com.censoft.censoftrongtong.controller;

import com.censoft.censoftrongtong.domain.*;
import com.censoft.censoftrongtong.domain.dto.*;
import com.censoft.censoftrongtong.enums.RiskPlanStatusType;
import com.censoft.censoftrongtong.service.*;
import com.ruoyi.common.config.RuoYiConfig;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.R;
import com.ruoyi.common.core.domain.entity.SysDictData;
import com.ruoyi.common.core.domain.entity.SysUser;
import com.ruoyi.system.service.ISysDictDataService;
import com.ruoyi.system.service.ISysUploadFileService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;


/**
 * 计划app端接口
 *
 * @author censoft
 * @date 2023-06-27
 * @module 融通项目
 */
@RestController
@RequestMapping("/app-api/risk/plan")
public class RiskPlanAppController extends BaseController {

    @Resource
    private ILedgerProjectService ledgerProjectService;

    @Resource
    private ILedgerBuildingService ledgerBuildingService;

    @Resource
    private ILedgerFloorService ledgerFloorService;

    @Resource
    private ILedgerRoomService ledgerRoomService;

    @Resource
    private IRiskPlanService riskPlanService;

    @Resource
    private IRiskPlanInherentListService riskPlanInherentListService;

    @Resource
    private IRiskPlanExistingListService riskPlanExistingListService;

    @Resource
    private ISysUploadFileService uploadFileService;

    @Resource
    private ILawListService lawListService;

    @Resource
    private ISysDictDataService dictDataService;

    @Resource
    private IRiskPlanUserBuildingService riskPlanUserBuildingService;

    /**
     * 服务器地址
     */
    @Value("${ruoyi.serverAddress}")
    private String serverAddress;

    /**
     * 获取自身部门及以下的项目
     *
     * @param ledgerProject 查询Dto
     * @real_return {@link R<List<LedgerProject>>}
     */
    @GetMapping("/project/list")
    public R<List<LedgerProject>> getProjectList(LedgerProject ledgerProject) {
        //添加自身及以下部门筛选
        ledgerProject.setDeptId(getLoginUser().getDeptId());
        return R.ok(ledgerProjectService.getProjectListByDeptId(ledgerProject));
    }

    /**
     * 获取项目下的楼宇
     *
     * @param projectId 项目id
     * @param type      类型 地上、地下
     * @real_return {@link R<List<LedgerBuilding>>}
     */
    @GetMapping("/building/list/{projectId}")
    public R<List<LedgerBuilding>> getBuildingListByProjectId(@PathVariable("projectId") Long projectId, String type) {
        return R.ok(ledgerBuildingService.getBuildingListByProjectId(projectId, type));
    }

    /**
     * 获取项目下的人员
     *
     * @param projectId 项目id
     * @real_return {@link R<List<SysUser>>}
     */
    @GetMapping("/user/list/{projectId}")
    public R<List<SysUser>> getUserListByProjectId(@PathVariable("projectId") Long projectId) {
        return R.ok(ledgerProjectService.getUserListByProjectId(projectId));
    }

    /**
     * 保存计划
     *
     * @param riskPlan 项目id
     * @real_return {@link R<List<LedgerProject>>}
     */
    @PostMapping()
    public R<Integer> saveRiskPlan(@RequestBody RiskPlan riskPlan) {
        riskPlan.setCreateUserId(getUserId());
        riskPlan.setCreateDeptId(getDeptId());
        return R.ok(riskPlanService.saveRiskPlan(riskPlan));
    }

    /**
     * 任务管理列表
     *
     * @param status 状态
     * @real_return {@link R<List<RiskPlanAppListDto>>}
     */
    @GetMapping("/create/list")
    public R<List<RiskPlanAppListDto>> getRiskPlanAppListDtoListByCreateUser(String status) {
        List<RiskPlanStatusType> statusList = new ArrayList<>();
        String state = "";
        if ("wait".equals(status)) {
            statusList = Arrays.asList(RiskPlanStatusType.CREATE_TASK,RiskPlanStatusType.WAIT_START);
            state = "待执行";
        } else if ("running".equals(status)) {
            statusList = Arrays.asList(RiskPlanStatusType.RUNNING,RiskPlanStatusType.WAIT_RETURN);
            state = "执行中";
        } else if ("finish".equals(status)) {
            statusList = Arrays.asList(RiskPlanStatusType.WAIT_LEADER_APPROVE,RiskPlanStatusType.WAIT_APPROVE
                    ,RiskPlanStatusType.PROJECT_REPORT,RiskPlanStatusType.FINISH);
            state = "已执行";
        }
        List<RiskPlanAppListDto> list = riskPlanService.getRiskPlanAppListDtoList(statusList,getUserId(),null ,null);
        String finalState = state;
        list.forEach(dto -> {
            dto.setState(finalState);
        });
        return R.ok(list);
    }

    /**
     * 任务执行情况列表
     *
     * @param status 状态 wait 待评估，running 执行中，finish 已完成
     * @real_return {@link R<List<RiskPlanAppListDto>>}
     */
    @GetMapping("/allocation/list")
    public R<List<RiskPlanAppListDto>> getAllocationRiskPlanAppListDtoListByCreateUser(String status) {
        List<RiskPlanStatusType> statusList = new ArrayList<>();
        String state = "";
        if ("wait".equals(status)) {
            statusList = Arrays.asList(RiskPlanStatusType.CREATE_TASK,RiskPlanStatusType.WAIT_START);
            state = "待评估";
        } else if ("running".equals(status)) {
            statusList = Arrays.asList(RiskPlanStatusType.RUNNING,RiskPlanStatusType.WAIT_RETURN);
            state = "执行中";
        } else if ("finish".equals(status)) {
            statusList = Arrays.asList(RiskPlanStatusType.WAIT_LEADER_APPROVE,RiskPlanStatusType.WAIT_APPROVE
                    ,RiskPlanStatusType.PROJECT_REPORT,RiskPlanStatusType.FINISH);
            state = "已完成";
        }
        List<RiskPlanAppListDto> list = riskPlanService.getRiskPlanAppListDtoList(statusList, getUserId(),null,null);
        String finalState = state;
        list.forEach(dto -> {
            Long finishNumber = riskPlanUserBuildingService.countStatusByPlanIdAndStatus(dto.getId(), "finish");
            Long totalNumber = riskPlanUserBuildingService.countStatusByPlanIdAndStatus(dto.getId(), null);

            dto.setState(finalState+"("+finishNumber+"/"+totalNumber+")");
        });
        return R.ok(list);
    }

    /**
     * 任务执行情况-楼栋详情列表
     *
     * @param planId 任务id
     * @real_return {@link R<List<RiskPlanAppListDto>>}
     */
    @GetMapping("/allocation/building/list/{planId}")
    public R<List<RiskPlanAppBuildingListDto>> getRiskPlanAppBuildingListDtoByPlanId(@PathVariable Long planId) {
        return R.ok(riskPlanUserBuildingService.getRiskPlanAppBuildingListDtoByPlanId(planId));
    }

    /**
     * 风险评估列表
     *
     * @param status 状态 running 待执行，finish 已执行，return 已退回
     * @real_return {@link R<List<RiskPlanAppListDto>>}
     */
    @GetMapping("/task/list")
    public R<List<RiskPlanAppListDto>> getRiskPlanAppListDtoListByLeaderUser(String status) {
        List<RiskPlanStatusType> statusList = new ArrayList<>();
        String state = "";
        if ("running".equals(status)) {
            statusList = Arrays.asList(RiskPlanStatusType.CREATE_TASK,RiskPlanStatusType.WAIT_START,RiskPlanStatusType.RUNNING);
            state = "待执行";
        } else if ("finish".equals(status)) {
            statusList = Arrays.asList(RiskPlanStatusType.WAIT_LEADER_APPROVE,RiskPlanStatusType.WAIT_APPROVE
                    ,RiskPlanStatusType.PROJECT_REPORT,RiskPlanStatusType.FINISH);
            state = "已执行";
        } else if ("return".equals(status)) {
            statusList = Arrays.asList(RiskPlanStatusType.WAIT_RETURN);
            state = "已退回";
        }

        List<RiskPlanAppListDto> list = riskPlanService.getRiskPlanAppListDtoList(statusList, null, null,getUserId());
        String finalState = state;
        list.forEach(dto -> {
            Long finishNumber = riskPlanUserBuildingService.countStatusByPlanIdAndStatus(dto.getId(), "finish");
            Long totalNumber = riskPlanUserBuildingService.countStatusByPlanIdAndStatus(dto.getId(), null);

            dto.setState(finalState+"("+finishNumber+"/"+totalNumber+")");
        });
        return R.ok(list);
    }

    /**
     * 获取任务单详情
     *
     * @param planId 计划id
     * @real_return {@link R<RiskPlanAppDetailsDto>}
     */
    @GetMapping("/details/{planId}")
    public R<RiskPlanAppDetailsDto> getRiskPlanAppDetailsDtoByPlanId(@PathVariable("planId") Long planId,Long buildingId) {
        return R.ok(riskPlanService.getRiskPlanAppDetailsDtoByPlanId(planId,buildingId));
    }

    /**
     * 获取计划下的楼宇
     *
     * @param planId 项目id
     * @real_return {@link R<List<LedgerBuilding>>}
     */
    @GetMapping("/plan/building/list/{planId}")
    public R<List<LedgerBuilding>> getBuildingListByPlanId(@PathVariable("planId") Long planId) {
        return R.ok(riskPlanService.getBuildingListByPlanId(planId));
    }

    /**
     * 根据楼宇id获取楼层列表
     *
     * @param buildingId 楼宇id
     * @real_return {@link R<List<LedgerFloor>>}
     */
    @GetMapping("/floor/list/{buildingId}")
    public R<List<LedgerFloor>> getFloorListByBuildingId(@PathVariable("buildingId") Long buildingId) {
        return R.ok(ledgerFloorService.getFloorListByBuildingId(buildingId));
    }

    /**
     * 根据楼层id获取房间列表
     *
     * @param floorId 楼宇id
     * @real_return {@link R<List<LedgerRoom>>}
     */
    @GetMapping("/room/list/{floorId}")
    public R<List<LedgerRoom>> getRoomListByBuildingId(@PathVariable("floorId") Long floorId) {
        return R.ok(ledgerRoomService.getRoomListByBuildingId(floorId));
    }

    /**
     * 根据计划id、楼宇id、楼层id、房间id获取固有风险列表
     *
     * @param planId     计划id
     * @param buildingId 楼宇id
     * @param floorId    楼层id
     * @param roomId     房间id
     * @real_return {@link R<List<RiskPlanAppInherentListDto>>}
     */
    @GetMapping("/inherent/list/{planId}/{buildingId}")
    public R<List<RiskPlanAppInherentListDto>> getInherentListByPlanId(@PathVariable("planId") Long planId
            , @PathVariable("buildingId") Long buildingId, Long floorId, Long roomId) {
        return R.ok(riskPlanInherentListService.getInherentListByPlanId(planId, buildingId, floorId, roomId));
    }

    /**
     * 添加固有风险
     *
     * @param saveDto 固有风险表单
     * @real_return {@link R<Integer>}
     */
    @PostMapping("/inherent")
    public R<Integer> saveRiskInherentListSaveDto(RiskInherentListSaveDto saveDto
            , @RequestParam(value = "pictureFile[]", required = false) MultipartFile[] pictureFile
            , @RequestParam(value = "measuresProjectFile[]", required = false) MultipartFile[] measuresProjectFile
            , @RequestParam(value = "measuresAdministrationFile[]", required = false) MultipartFile[] measuresAdministrationFile
            , @RequestParam(value = "measuresEmergencyFile[]", required = false) MultipartFile[] measuresEmergencyFile) {

        String userId = getUserId().toString();

        //风险源照片
        if (pictureFile != null) {
            String pictureFileIds = Arrays.stream(pictureFile)
                    .filter(multipartFile -> !multipartFile.isEmpty())
                    .map(multipartFile -> uploadFileService.upload(multipartFile, RuoYiConfig.getUploadPath(), userId, serverAddress).toString())
                    .collect(Collectors.joining(","));
            saveDto.setPictureFileIds(pictureFileIds);
        }

        //工程技术措施附件
        if (measuresProjectFile != null) {
            String measuresProjectFileIds = Arrays.stream(measuresProjectFile)
                    .filter(multipartFile -> !multipartFile.isEmpty())
                    .map(multipartFile -> uploadFileService.upload(multipartFile, RuoYiConfig.getUploadPath(), userId, serverAddress).toString())
                    .collect(Collectors.joining(","));
            saveDto.setMeasuresProjectFileIds(measuresProjectFileIds);
        }

        //管理措施附件
        if (measuresAdministrationFile != null) {
            String measuresAdministrationFileIds = Arrays.stream(measuresAdministrationFile)
                    .filter(multipartFile -> !multipartFile.isEmpty())
                    .map(multipartFile -> uploadFileService.upload(multipartFile, RuoYiConfig.getUploadPath(), userId, serverAddress).toString())
                    .collect(Collectors.joining(","));
            saveDto.setMeasuresAdministrationFileIds(measuresAdministrationFileIds);
        }

        //应急处置措施附件
        if (measuresEmergencyFile != null) {
            String measuresEmergencyFileIds = Arrays.stream(measuresEmergencyFile)
                    .filter(multipartFile -> !multipartFile.isEmpty())
                    .map(multipartFile -> uploadFileService.upload(multipartFile, RuoYiConfig.getUploadPath(), userId, serverAddress).toString())
                    .collect(Collectors.joining(","));
            saveDto.setMeasuresEmergencyFileIds(measuresEmergencyFileIds);
        }

        return R.ok(riskPlanInherentListService.saveRiskInherentListSaveDto(saveDto));
    }

    /**
     * 获取固有风险详情
     *
     * @param inherentId 固有风险id
     * @real_return {@link R<RiskPlanAppInherentListDetailsDto>}
     */
    @GetMapping("/inherent/details/{inherentId}")
    public R<RiskPlanAppInherentListDetailsDto> getRiskPlanAppInherentListDetailsDtoByInherentId(@PathVariable("inherentId") Long inherentId) {
        return R.ok(riskPlanInherentListService.getRiskPlanAppInherentListDetailsDtoByInherentId(inherentId));
    }

    /**
     * 根据固有风险id获取现有风险列表
     *
     * @param inherentId 固有风险id
     * @real_return {@link R<List<RiskPlanAppExistingListDto>>}
     */
    @GetMapping("/existing/list/{inherentId}")
    public R<List<RiskPlanAppExistingListDto>> getExistingListByInherentId(@PathVariable("inherentId") Long inherentId) {
        return R.ok(riskPlanInherentListService.getExistingListByInherentId(inherentId));
    }


    /**
     * 添加现有风险
     *
     * @param saveDto 现有风险表单
     * @real_return {@link R<Integer>}
     */
    @PostMapping("/existing")
    public R<Integer> saveRiskExistingListSaveDto(RiskExistingListSaveDto saveDto
            , @RequestParam(value = "pictureFile[]", required = false) MultipartFile[] pictureFile
            , @RequestParam(value = "measuresProjectFile[]", required = false) MultipartFile[] measuresProjectFile
            , @RequestParam(value = "measuresAdministrationFile[]", required = false) MultipartFile[] measuresAdministrationFile
            , @RequestParam(value = "measuresEmergencyFile[]", required = false) MultipartFile[] measuresEmergencyFile) {

        String userId = getUserId().toString();

        //风险源照片
        if (pictureFile != null) {
            String pictureFileIds = Arrays.stream(pictureFile)
                    .filter(multipartFile -> !multipartFile.isEmpty())
                    .map(multipartFile -> uploadFileService.upload(multipartFile, RuoYiConfig.getUploadPath(), userId, serverAddress).toString())
                    .collect(Collectors.joining(","));
            saveDto.setPictureFileIds(pictureFileIds);
        }

        //工程技术措施附件
        if (measuresProjectFile != null) {
            String measuresProjectFileIds = Arrays.stream(measuresProjectFile)
                    .filter(multipartFile -> !multipartFile.isEmpty())
                    .map(multipartFile -> uploadFileService.upload(multipartFile, RuoYiConfig.getUploadPath(), userId, serverAddress).toString())
                    .collect(Collectors.joining(","));
            saveDto.setMeasuresProjectFileIds(measuresProjectFileIds);
        }

        //管理措施附件
        if (measuresAdministrationFile != null) {
            String measuresAdministrationFileIds = Arrays.stream(measuresAdministrationFile)
                    .filter(multipartFile -> !multipartFile.isEmpty())
                    .map(multipartFile -> uploadFileService.upload(multipartFile, RuoYiConfig.getUploadPath(), userId, serverAddress).toString())
                    .collect(Collectors.joining(","));
            saveDto.setMeasuresAdministrationFileIds(measuresAdministrationFileIds);
        }

        //应急处置措施附件
        if (measuresEmergencyFile != null) {
            String measuresEmergencyFileIds = Arrays.stream(measuresEmergencyFile)
                    .filter(multipartFile -> !multipartFile.isEmpty())
                    .map(multipartFile -> uploadFileService.upload(multipartFile, RuoYiConfig.getUploadPath(), userId, serverAddress).toString())
                    .collect(Collectors.joining(","));
            saveDto.setMeasuresEmergencyFileIds(measuresEmergencyFileIds);
        }

        return R.ok(riskPlanExistingListService.saveRiskExistingListSaveDto(saveDto));
    }

    /**
     * 风险评估-任务完成
     *
     * @param planId 任务id
     * @param buildingId 楼宇id
     * @real_return {@link R<Integer>}
     */
    @PostMapping("/submit/{planId}/{buildingId}")
    public R<Integer> submitByPlanIdAndBuildingId(@PathVariable Long planId, @PathVariable Long buildingId) {
        return R.ok(riskPlanUserBuildingService.submitByPlanIdAndBuildingId(planId,buildingId, getUserId()));
    }


    /**
     * 根据风险任务审批列表
     *
     * @param status 状态 running 待审批 completed 已审批
     * @real_return {@link R<List<RiskPlanAppExistingListDto>>}
     */
    @GetMapping("/approve/list")
    public R<List<RiskPlanAppListDto>> getRiskPlanAppListDtoListByApproveUser(String status) {
        return R.ok(riskPlanService.getRiskPlanAppListDtoListByApproveUser(getUserId(),status));
    }


    /**
     * 根据风险任务审批
     *
     * @real_return {@link R<Integer>}
     */
    @PostMapping("/approve")
    public R<Integer> approveAppApproveDto(@RequestBody RiskPlanAppApproveDto appApproveDto) {
        appApproveDto.setUserId(getUserId());
        return R.ok(riskPlanService.approveAppApproveDto(appApproveDto));
    }


    /**
     * 退回删除接口
     *
     * @real_return {@link R<Integer>}
     */
    @PostMapping("/revoke/{planId}")
    public R<Integer> revokeByPlanId(@PathVariable("planId") Long planId) {
        return R.ok(riskPlanService.revokeByPlanId(planId,getUserId()));
    }


    /**
     * 获取法律依据
     *
     * @real_return {@link R<List<LawList>>}
     */
    @GetMapping("/law/list")
    public R<List<LawList>> getLawList(LawList lawList) {
        return R.ok(lawListService.selectLawListList(lawList));
    }


    /**
     * 根据字典key获取字典列表
     *
     * @real_return {@link R<List<SysDictData>>}
     */
    @GetMapping("/dict/data/list")
    public R<List<SysDictData>> getSysDictDataList(SysDictData dictData) {
        return R.ok(dictDataService.selectDictDataList(dictData));
    }
}
