package com.censoft.censoftrongtong.service.impl;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

import com.censoft.censoftrongtong.domain.LedgerBuilding;
import com.censoft.censoftrongtong.domain.RiskPlan;
import com.censoft.censoftrongtong.domain.RiskPlanTask;
import com.censoft.censoftrongtong.domain.RiskPlanUserBuilding;
import com.censoft.censoftrongtong.domain.dto.RiskPlanAppApproveDto;
import com.censoft.censoftrongtong.domain.dto.RiskPlanAppDetailsDto;
import com.censoft.censoftrongtong.domain.dto.RiskPlanAppListDto;
import com.censoft.censoftrongtong.enums.RiskPlanStatusType;
import com.censoft.censoftrongtong.mapper.RiskPlanMapper;
import com.censoft.censoftrongtong.service.ILedgerBuildingService;
import com.censoft.censoftrongtong.service.IRiskPlanService;
import com.censoft.censoftrongtong.service.IRiskPlanTaskService;
import com.censoft.censoftrongtong.service.IRiskPlanUserBuildingService;
import com.ruoyi.common.utils.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

/**
 * 风险计划Service业务层处理
 *
 * @author ruoyi
 * @date 2023-06-26
 */
@Service
public class RiskPlanServiceImpl implements IRiskPlanService {
    @Autowired
    private RiskPlanMapper riskPlanMapper;

    @Resource
    private IRiskPlanTaskService riskPlanTaskService;

    @Resource
    private ILedgerBuildingService ledgerBuildingService;

    @Resource
    private IRiskPlanUserBuildingService riskPlanUserBuildingService;

    /**
     * 查询风险计划
     *
     * @param id 风险计划主键
     * @return 风险计划
     */
    @Override
    public RiskPlan selectRiskPlanById(Long id) {
        return riskPlanMapper.selectRiskPlanById(id);
    }

    /**
     * 查询风险计划列表
     *
     * @param riskPlan 风险计划
     * @return 风险计划
     */
    @Override
    public List<RiskPlan> selectRiskPlanList(RiskPlan riskPlan) {
        return riskPlanMapper.selectRiskPlanList(riskPlan);
    }

    /**
     * 新增风险计划
     *
     * @param riskPlan 风险计划
     * @return 结果
     */
    @Override
    public int insertRiskPlan(RiskPlan riskPlan) {
        riskPlan.setCreateTime(DateUtils.getNowDate());
        return riskPlanMapper.insertRiskPlan(riskPlan);
    }

    /**
     * 修改风险计划
     *
     * @param riskPlan 风险计划
     * @return 结果
     */
    @Override
    public int updateRiskPlan(RiskPlan riskPlan) {
        riskPlan.setUpdateTime(DateUtils.getNowDate());
        return riskPlanMapper.updateRiskPlan(riskPlan);
    }

    /**
     * 批量删除风险计划
     *
     * @param ids 需要删除的风险计划主键
     * @return 结果
     */
    @Override
    public int deleteRiskPlanByIds(Long[] ids) {
        return riskPlanMapper.deleteRiskPlanByIds(ids);
    }

    /**
     * 删除风险计划信息
     *
     * @param id 风险计划主键
     * @return 结果
     */
    @Override
    public int deleteRiskPlanById(Long id) {
        return riskPlanMapper.deleteRiskPlanById(id);
    }

    @Override
    public int saveRiskPlan(RiskPlan riskPlan) {
        //插入主表
        riskPlan.setNo(getNo());
        riskPlan.setStatus(RiskPlanStatusType.WAIT_START.getStatus());
        int i = insertRiskPlan(riskPlan);

        //插入任务表 已完成任务发起
        RiskPlanTask riskPlanTask = new RiskPlanTask();
        riskPlanTask.setPlanId(riskPlan.getId());
        riskPlanTask.setTaskKey(RiskPlanStatusType.CREATE_TASK.getStatus());
        riskPlanTask.setTaskName(RiskPlanStatusType.CREATE_TASK.getTitle());
        riskPlanTask.setStatus("completed");
        riskPlanTask.setStartTime(new Date());
        riskPlanTask.setCompletedTime(new Date());
        riskPlanTaskService.insertRiskPlanTask(riskPlanTask);

        //插入任务表 待执行风险评估
        RiskPlanTask riskPlanTask1 = new RiskPlanTask();
        riskPlanTask1.setPlanId(riskPlan.getId());
        riskPlanTask1.setApprovalUserId(riskPlan.getLeaderUserId());
        riskPlanTask1.setTaskKey(RiskPlanStatusType.WAIT_START.getStatus());
        riskPlanTask1.setTaskName(RiskPlanStatusType.WAIT_START.getTitle());
        riskPlanTask1.setStatus("running");
        riskPlanTask1.setStartTime(new Date());
        riskPlanTaskService.insertRiskPlanTask(riskPlanTask1);

        //插入计划人员与楼宇关系表
        List<RiskPlanUserBuilding> list = Arrays.stream(riskPlan.getBuildingIds().split(",")).map(buildingId -> {
            RiskPlanUserBuilding riskPlanUserBuilding = new RiskPlanUserBuilding();
            riskPlanUserBuilding.setPlanId(riskPlan.getId());
            riskPlanUserBuilding.setBuildingId(Long.valueOf(buildingId));
            riskPlanUserBuilding.setUserIds(riskPlan.getWorkUserIds());
            riskPlanUserBuilding.setStartTime(new Date());
            riskPlanUserBuilding.setStatus("wait");
            return riskPlanUserBuilding;
        }).collect(Collectors.toList());
        riskPlanUserBuildingService.saveBatch(list);

        return i;
    }

    @Override
    public List<RiskPlanAppListDto> getRiskPlanAppListDtoList(List<RiskPlanStatusType> statusList, Long createUserId, Long leaderUserId) {
        List<String> statusStringList = null;
        //枚举类转化为数据库存的状态值
        if (statusList != null) {
            statusStringList = statusList.stream()
                    .map(RiskPlanStatusType::getStatus)
                    .collect(Collectors.toList());
        }

        //查询并把状态值转化为状态名称
        return riskPlanMapper.getRiskPlanAppListDtoList(statusStringList, createUserId, leaderUserId).stream().map(dto -> {
            String status = RiskPlanStatusType.getTitleByStatus(dto.getStatus()).getTitle();
            dto.setStatus(status);
            return dto;
        }).collect(Collectors.toList());
    }

    @Override
    public RiskPlanAppDetailsDto getRiskPlanAppDetailsDtoByPlanId(Long planId) {
        //获取任务单基础信息
        RiskPlanAppDetailsDto dto = riskPlanMapper.getRiskPlanAppDetailsDtoByPlanId(planId);
        dto.setStatus(RiskPlanStatusType.getTitleByStatus(dto.getStatusKey()).getTitle());

        //获取任务详情
        dto.setDetailsDto(riskPlanMapper.getRiskPlanAppTaskDetailsDtoByPlanId(planId));

        //风险清单
        dto.setRiskBuildingListDtos(riskPlanMapper.getRiskBuildingListDtoByPlanId(planId));

        //审批记录
        dto.setApproveListDtos(riskPlanMapper.getApproveListDtoByPlanId(planId));

        //风险评估报告
        dto.setFiles(new ArrayList());

        return dto;
    }

    @Override
    public List<LedgerBuilding> getBuildingListByPlanId(Long planId) {
        RiskPlan riskPlan = selectRiskPlanById(planId);
        return ledgerBuildingService.getBuildingListByIds(Arrays.asList(riskPlan.getBuildingIds().split(",")));
    }

    @Override
    public int approveAppApproveDto(RiskPlanAppApproveDto appApproveDto) {
        //获取最新的task
        RiskPlanTask lastRiskPlanTask = riskPlanTaskService.getLastRiskPlanTaskByPlanId(appApproveDto.getId());
        lastRiskPlanTask.setTaskResult(appApproveDto.getTaskResult());
        lastRiskPlanTask.setTaskOpinion(appApproveDto.getTaskOpinion());
        lastRiskPlanTask.setStatus("completed");
        lastRiskPlanTask.setCompletedTime(new Date());
        riskPlanTaskService.updateRiskPlanTask(lastRiskPlanTask);

        //计划
        RiskPlan riskPlan = selectRiskPlanById(lastRiskPlanTask.getPlanId());

        if ("0".equals(appApproveDto.getTaskResult())) {
            //退回
            //插入任务表 已退回风险评估
            RiskPlanTask riskPlanTask1 = new RiskPlanTask();
            riskPlanTask1.setPlanId(lastRiskPlanTask.getPlanId());
            riskPlanTask1.setApprovalUserId(riskPlan.getLeaderUserId());
            riskPlanTask1.setTaskKey(RiskPlanStatusType.WAIT_RETURN.getStatus());
            riskPlanTask1.setTaskName(RiskPlanStatusType.WAIT_RETURN.getStatus());
            riskPlanTask1.setStatus("running");
            riskPlanTask1.setStartTime(new Date());
            riskPlanTaskService.insertRiskPlanTask(riskPlanTask1);

            //修改计划状态
            riskPlan.setStatus(RiskPlanStatusType.WAIT_RETURN.getStatus());
            updateRiskPlan(riskPlan);
        } else {
            //修改计划状态
            riskPlan.setStatus(RiskPlanStatusType.FINISH.getStatus());
            updateRiskPlan(riskPlan);
        }
        return 1;
    }

    @Override
    public int revokeByPlanId(Long planId, Long userId) {
        //计划
        RiskPlan riskPlan = selectRiskPlanById(planId);

        //获取最新的task
        RiskPlanTask lastRiskPlanTask = riskPlanTaskService.getLastRiskPlanTaskByPlanId(planId);
        lastRiskPlanTask.setStatus("completed");
        lastRiskPlanTask.setCompletedTime(new Date());
        riskPlanTaskService.updateRiskPlanTask(lastRiskPlanTask);

        //插入任务表 已撤销风险评估
        RiskPlanTask riskPlanTask1 = new RiskPlanTask();
        riskPlanTask1.setPlanId(lastRiskPlanTask.getPlanId());
        riskPlanTask1.setTaskKey(RiskPlanStatusType.REVOKE.getStatus());
        riskPlanTask1.setTaskName(RiskPlanStatusType.REVOKE.getStatus());
        riskPlanTask1.setStatus("completed");
        riskPlanTask1.setStartTime(new Date());
        riskPlanTaskService.insertRiskPlanTask(riskPlanTask1);

        //修改计划状态
        riskPlan.setStatus(RiskPlanStatusType.REVOKE.getStatus());
        updateRiskPlan(riskPlan);

        return 1;
    }

    private String getNo() {
        Random random = new Random();
        int randomNumber = random.nextInt(1000); // 生成0到999之间的随机数
        String formattedNumber = String.format("%03d", randomNumber); // 格式化为三位数，不足三位前面补0
        return DateUtils.dateTimeNow() + formattedNumber;
    }
}
