package com.censoft.censoftrongtong.controller;

import com.censoft.censoftrongtong.domain.LedgerBuilding;
import com.censoft.censoftrongtong.domain.LedgerProject;
import com.censoft.censoftrongtong.domain.RiskPlan;
import com.censoft.censoftrongtong.domain.dto.RiskPlanAppListDto;
import com.censoft.censoftrongtong.enums.RiskPlanStatusType;
import com.censoft.censoftrongtong.service.ILedgerBuildingService;
import com.censoft.censoftrongtong.service.ILedgerProjectService;
import com.censoft.censoftrongtong.service.IRiskPlanService;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.core.domain.R;
import com.ruoyi.common.core.domain.entity.SysUser;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.Arrays;
import java.util.List;


/**
 * 计划app端接口
 *
 * @author censoft
 * @date 2023-06-27
 * @module 融通项目
 */
@RestController
@RequestMapping("/app-api/risk/plan")
public class RiskPlanAppController extends BaseController {

    @Resource
    private ILedgerProjectService ledgerProjectService;

    @Resource
    private ILedgerBuildingService ledgerBuildingService;

    @Resource
    private IRiskPlanService riskPlanService;

    /**
     * 获取自身部门及以下的项目
     *
     * @param ledgerProject 查询Dto
     * @real_return {@link R<List<LedgerProject>>}
     */
    @GetMapping("/project/list")
    public R<List<LedgerProject>> getProjectList(LedgerProject ledgerProject){
        //添加自身及以下部门筛选
        ledgerProject.setDeptId(getLoginUser().getDeptId());
        return R.ok(ledgerProjectService.getProjectListByDeptId(ledgerProject));
    }

    /**
     * 获取项目下的楼宇
     *
     * @param projectId 项目id
     * @real_return {@link R<List<LedgerBuilding>>}
     */
    @GetMapping("/building/list/{projectId}")
    public R<List<LedgerBuilding>> getBuildingListByProjectId(@PathVariable Long projectId){
        return R.ok(ledgerBuildingService.getBuildingListByProjectId(projectId));
    }

    /**
     * 获取项目下的人员
     *
     * @param projectId 项目id
     * @real_return {@link R<List<SysUser>>}
     */
    @GetMapping("/user/list/{projectId}")
    public R<List<SysUser>> getUserListByProjectId(@PathVariable Long projectId){
        return R.ok(ledgerProjectService.getUserListByProjectId(projectId));
    }

    /**
     * 保存计划
     *
     * @param riskPlan 项目id
     * @real_return {@link R<List<LedgerProject>>}
     */
    @PostMapping()
    public R<Integer> saveRiskPlan(@RequestBody RiskPlan riskPlan){
        riskPlan.setCreateUserId(getUserId());
        riskPlan.setCreateDeptId(getDeptId());
        return R.ok(riskPlanService.saveRiskPlan(riskPlan));
    }

    /**
     * 任务管理列表
     *
     * @param status 状态
     * @real_return {@link R<List<RiskPlanAppListDto>>}
     */
    @GetMapping("/create/list")
    public R<List<RiskPlanAppListDto>> getRiskPlanAppListDtoListByCreateUser(String status){
        List<RiskPlanStatusType> statusList;
        if ("running".equals(status)){
            statusList = Arrays.asList(RiskPlanStatusType.WAIT_START,RiskPlanStatusType.RUNNING);
        }else {
            statusList = Arrays.asList(RiskPlanStatusType.WAIT_FINISH,RiskPlanStatusType.FINISH
                    ,RiskPlanStatusType.WAIT_ADJUST,RiskPlanStatusType.REVOKE);
        }
        return R.ok(riskPlanService.getRiskPlanAppListDtoList(statusList,getUserId(),null));
    }

    /**
     * 风险评估列表
     *
     * @real_return {@link R<List<RiskPlanAppListDto>>}
     */
    @GetMapping("/task/list")
    public R<List<RiskPlanAppListDto>> getRiskPlanAppListDtoListByLeaderUser(){
        return R.ok(riskPlanService.getRiskPlanAppListDtoList(null,null,getUserId()));
    }
}
