package com.ruoyi.algorithm.service.impl;

import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

import cn.hutool.core.util.StrUtil;
import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.ruoyi.algorithm.domain.AlgorithmScenePiece;
import com.ruoyi.algorithm.domain.dto.AlgorithmPieceDetailsDto;
import com.ruoyi.algorithm.domain.dto.AlgorithmSceneDetailsDto;
import com.ruoyi.algorithm.domain.dto.AlgorithmSceneListDto;
import com.ruoyi.algorithm.domain.dto.AlgorithmSceneSaveDto;
import com.ruoyi.algorithm.service.IAlgorithmScenePieceService;
import com.ruoyi.algorithm.service.IAlgorithmSceneStatusService;
import com.ruoyi.algorithm.utils.FlinkRestApiUtil;
import com.ruoyi.common.config.AiRestApiConfig;
import com.ruoyi.common.config.FlinkConfig;
import com.ruoyi.common.exception.ServiceException;
import com.ruoyi.common.utils.bean.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.ruoyi.algorithm.mapper.AlgorithmSceneBaseMapper;
import com.ruoyi.algorithm.domain.AlgorithmSceneBase;
import com.ruoyi.algorithm.service.IAlgorithmSceneBaseService;

import javax.annotation.Resource;

/**
 * 算法场景Service业务层处理
 *
 * @author ruoyi
 * @date 2023-04-26
 */
@Service
public class AlgorithmSceneBaseServiceImpl implements IAlgorithmSceneBaseService {
    @Autowired
    private AlgorithmSceneBaseMapper algorithmSceneBaseMapper;

    @Resource
    private IAlgorithmSceneStatusService algorithmSceneStatusService;

    @Resource
    private IAlgorithmScenePieceService algorithmScenePieceService;

    /**
     * 查询算法场景
     *
     * @param id 算法场景主键
     * @return 算法场景
     */
    @Override
    public AlgorithmSceneBase selectAlgorithmSceneBaseById(Long id) {
        return algorithmSceneBaseMapper.selectAlgorithmSceneBaseById(id);
    }

    /**
     * 查询算法场景列表
     *
     * @param algorithmSceneBase 算法场景
     * @return 算法场景
     */
    @Override
    public List<AlgorithmSceneBase> selectAlgorithmSceneBaseList(AlgorithmSceneBase algorithmSceneBase) {
        return algorithmSceneBaseMapper.selectAlgorithmSceneBaseList(algorithmSceneBase);
    }

    /**
     * 新增算法场景
     *
     * @param algorithmSceneBase 算法场景
     * @return 结果
     */
    @Override
    public int insertAlgorithmSceneBase(AlgorithmSceneBase algorithmSceneBase) {
        return algorithmSceneBaseMapper.insertAlgorithmSceneBase(algorithmSceneBase);
    }

    /**
     * 修改算法场景
     *
     * @param algorithmSceneBase 算法场景
     * @return 结果
     */
    @Override
    public int updateAlgorithmSceneBase(AlgorithmSceneBase algorithmSceneBase) {
        return algorithmSceneBaseMapper.updateAlgorithmSceneBase(algorithmSceneBase);
    }

    /**
     * 批量删除算法场景
     *
     * @param ids 需要删除的算法场景主键
     * @return 结果
     */
    @Override
    public int deleteAlgorithmSceneBaseByIds(Long[] ids) {
        return algorithmSceneBaseMapper.deleteAlgorithmSceneBaseByIds(ids);
    }

    /**
     * 删除算法场景信息
     *
     * @param id 算法场景主键
     * @return 结果
     */
    @Override
    public int deleteAlgorithmSceneBaseById(Long id) {
        return algorithmSceneBaseMapper.deleteAlgorithmSceneBaseById(id);
    }

    @Override
    public List<AlgorithmSceneListDto> selectAlgorithmSceneListDto(AlgorithmSceneBase algorithmSceneBase) {
        return algorithmSceneBaseMapper.selectAlgorithmSceneListDto(algorithmSceneBase);
    }

    @Override
    public int editStatus(AlgorithmSceneBase algorithmSceneBase) {
        //判断状态是否正确
        AlgorithmSceneBase sceneBase = selectAlgorithmSceneBaseById(algorithmSceneBase.getId());
        if (sceneBase.getStatus().equals(algorithmSceneBase.getStatus())) {
            throw new ServiceException("场景状态错误，请刷新后再试！");
        }

        //通知算法启动停止
        sendAi(algorithmSceneBase.getId(), algorithmSceneBase.getStatus());

        //启动、取消flink流处理
        AlgorithmSceneBase newSceneBase = new AlgorithmSceneBase();
        newSceneBase.setId(algorithmSceneBase.getId());
        newSceneBase.setStatus(algorithmSceneBase.getStatus());
        newSceneBase.setJobId("");
        if ("0".equals(algorithmSceneBase.getStatus())) {
            // flink 启动job
            String result = FlinkRestApiUtil.runJar(FlinkConfig.getJarId(), FlinkConfig.getEntryClass()
                    , StrUtil.format(FlinkConfig.getProgramArgs(), algorithmSceneBase.getId()));
            JSONObject jsonObject = JSONUtil.parseObj(result);
            String jobId = jsonObject.getStr("job");
            newSceneBase.setJobId(jobId);
        } else {
            // flink 取消job
            String s = FlinkRestApiUtil.cancelJob(sceneBase.getJobId());
        }
        //修改场景状态
        updateAlgorithmSceneBase(newSceneBase);
        return 1;
    }

    @Override
    public AlgorithmSceneDetailsDto selectAlgorithmSceneDetailsDtoById(Long id) {
        AlgorithmSceneBase algorithmSceneBase = algorithmSceneBaseMapper.selectAlgorithmSceneBaseById(id);
        AlgorithmSceneDetailsDto algorithmSceneDetailsDto = new AlgorithmSceneDetailsDto();
        BeanUtils.copyBeanProp(algorithmSceneDetailsDto, algorithmSceneBase);
        algorithmSceneDetailsDto.setPieceDtos(algorithmScenePieceService.selectAlgorithmPieceDetailsDtoBySceneId(id));
        return algorithmSceneDetailsDto;
    }

    @Override
    public int insertAlgorithmSceneSaveDto(AlgorithmSceneSaveDto algorithmSceneSaveDto) {
        //新建场景
        AlgorithmSceneBase algorithmSceneBase = new AlgorithmSceneBase();
        BeanUtils.copyBeanProp(algorithmSceneBase, algorithmSceneSaveDto);
        algorithmSceneBase.setAlarmTypes(algorithmSceneSaveDto.getAlarmTypes().stream().collect(Collectors.joining(",")));
        algorithmSceneBase.setCameraIdList(algorithmSceneSaveDto.getCameraIdList().stream().collect(Collectors.joining(",")));
        int i = insertAlgorithmSceneBase(algorithmSceneBase);

        //删除场景下算法块及参数数据
        algorithmScenePieceService.deleteAlgorithmScenePieceBySceneId(algorithmSceneBase.getId());

        //新建场景下算法块及参数
        List<AlgorithmScenePiece> algorithmScenePieceList = algorithmSceneSaveDto
                .getPieceDtos()
                .stream()
                .flatMap(pieceDto -> pieceDto.getVariables()
                        .stream()
                        .map(variable -> {
                            AlgorithmScenePiece algorithmScenePiece = new AlgorithmScenePiece();
                            algorithmScenePiece.setSceneId(algorithmSceneBase.getId());
                            algorithmScenePiece.setPieceId(pieceDto.getId());
                            algorithmScenePiece.setVariableKey(variable.getVariableKey());
                            algorithmScenePiece.setVariableValue(variable.getVariableValue());
                            algorithmScenePiece.setSort(pieceDto.getSort());
                            return algorithmScenePiece;
                        })).collect(Collectors.toList());

        for (AlgorithmScenePiece algorithmScenePiece : algorithmScenePieceList) {
            algorithmScenePieceService.insertAlgorithmScenePiece(algorithmScenePiece);
        }

        return i;
    }

    @Override
    public int updateAlgorithmSceneSaveDto(AlgorithmSceneSaveDto algorithmSceneSaveDto) {
        //更新场景
        AlgorithmSceneBase algorithmSceneBase = new AlgorithmSceneBase();
        BeanUtils.copyBeanProp(algorithmSceneBase, algorithmSceneSaveDto);
        algorithmSceneBase.setAlarmTypes(algorithmSceneSaveDto.getAlarmTypes().stream().collect(Collectors.joining(",")));
        algorithmSceneBase.setCameraIdList(algorithmSceneSaveDto.getCameraIdList().stream().collect(Collectors.joining(",")));
        int i = updateAlgorithmSceneBase(algorithmSceneBase);

        //删除场景下算法块及参数数据
        algorithmScenePieceService.deleteAlgorithmScenePieceBySceneId(algorithmSceneBase.getId());

        //新建场景下算法块及参数
        List<AlgorithmScenePiece> algorithmScenePieceList = algorithmSceneSaveDto
                .getPieceDtos()
                .stream()
                .flatMap(pieceDto -> pieceDto.getVariables()
                        .stream()
                        .map(variable -> {
                            AlgorithmScenePiece algorithmScenePiece = new AlgorithmScenePiece();
                            algorithmScenePiece.setSceneId(algorithmSceneBase.getId());
                            algorithmScenePiece.setPieceId(pieceDto.getId());
                            algorithmScenePiece.setVariableKey(variable.getVariableKey());
                            algorithmScenePiece.setVariableValue(variable.getVariableValue());
                            algorithmScenePiece.setSort(pieceDto.getSort());
                            return algorithmScenePiece;
                        })).collect(Collectors.toList());

        for (AlgorithmScenePiece algorithmScenePiece : algorithmScenePieceList) {
            algorithmScenePieceService.insertAlgorithmScenePiece(algorithmScenePiece);
        }

        return i;
    }

    @Override
    public int copyById(Long id, String username) {
        //更新场景
        AlgorithmSceneBase oldAlgorithmSceneBase = selectAlgorithmSceneBaseById(id);
        AlgorithmSceneBase newAlgorithmSceneBase = new AlgorithmSceneBase();
        BeanUtils.copyBeanProp(newAlgorithmSceneBase, oldAlgorithmSceneBase);
        newAlgorithmSceneBase.setStatus("1");
        newAlgorithmSceneBase.setCreateBy(username);
        newAlgorithmSceneBase.setSceneName(oldAlgorithmSceneBase.getSceneName()+"（副本）");
        int i = insertAlgorithmSceneBase(newAlgorithmSceneBase);

        //删除场景下算法块及参数数据
        algorithmScenePieceService.deleteAlgorithmScenePieceBySceneId(newAlgorithmSceneBase.getId());

        //查找旧场景下算法
        List<AlgorithmPieceDetailsDto> algorithmPieceDetailsDtos = algorithmScenePieceService.selectAlgorithmPieceDetailsDtoBySceneId(id);


        //新建场景下算法块及参数
        List<AlgorithmScenePiece> algorithmScenePieceList = algorithmPieceDetailsDtos
                .stream()
                .flatMap(pieceDto -> pieceDto.getVariables()
                        .stream()
                        .map(variable -> {
                            AlgorithmScenePiece algorithmScenePiece = new AlgorithmScenePiece();
                            algorithmScenePiece.setSceneId(newAlgorithmSceneBase.getId());
                            algorithmScenePiece.setPieceId(pieceDto.getId());
                            algorithmScenePiece.setVariableKey(variable.getVariableKey());
                            algorithmScenePiece.setVariableValue(variable.getVariableValue());
                            algorithmScenePiece.setSort(pieceDto.getSort());
                            return algorithmScenePiece;
                        })).collect(Collectors.toList());

        for (AlgorithmScenePiece algorithmScenePiece : algorithmScenePieceList) {
            algorithmScenePieceService.insertAlgorithmScenePiece(algorithmScenePiece);
        }

        return i;
    }

    private void sendAi(Long sceneId, String status) {
        String statusKey;
        //启动
        if ("0".equals(status)) {
            //生成ai表数据
            algorithmSceneStatusService.createBySceneId(sceneId);
            statusKey = "run";
        } else {
            //停止，删除ai表数据
            algorithmSceneStatusService.deleteBySceneId(sceneId);
            statusKey = "stop";
        }

        //通知ai算法
        String url = "http://{}:{}/{}?sceneId={}";
        url = StrUtil.format(url, AiRestApiConfig.getHost(), AiRestApiConfig.getPort(), statusKey, sceneId);
        HttpUtil.post(url, new HashMap<>());
        return;
    }
}
