package com.ruoyi.algorithm.controller;

import java.util.List;
import javax.servlet.http.HttpServletResponse;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.algorithm.domain.AlgorithmBaseType;
import com.ruoyi.algorithm.service.IAlgorithmBaseTypeService;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;

/**
 * 算法预警类型Controller
 * 
 * @author ruoyi
 * @date 2023-04-26
 */
@RestController
@RequestMapping("/system/algorithm/type")
public class AlgorithmBaseTypeController extends BaseController
{
    @Autowired
    private IAlgorithmBaseTypeService algorithmBaseTypeService;

    /**
     * 查询算法预警类型列表
     */
    @PreAuthorize("@ss.hasPermi('system:type:list')")
    @GetMapping("/list")
    public TableDataInfo list(AlgorithmBaseType algorithmBaseType)
    {
        startPage();
        List<AlgorithmBaseType> list = algorithmBaseTypeService.selectAlgorithmBaseTypeList(algorithmBaseType);
        return getDataTable(list);
    }

    /**
     * 查询算法预警类型列表
     */
    @PreAuthorize("@ss.hasPermi('system:type:list')")
    @GetMapping("/list/{algorithmId}")
    public AjaxResult listByAlgorithmBaseId(@PathVariable Long algorithmId)
    {
        AlgorithmBaseType algorithmBaseType = new AlgorithmBaseType();
        algorithmBaseType.setAlgorithmId(algorithmId);
        return success(algorithmBaseTypeService.selectAlgorithmBaseTypeList(algorithmBaseType));
    }

    /**
     * 导出算法预警类型列表
     */
    @PreAuthorize("@ss.hasPermi('system:type:export')")
    @Log(title = "算法预警类型", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, AlgorithmBaseType algorithmBaseType)
    {
        List<AlgorithmBaseType> list = algorithmBaseTypeService.selectAlgorithmBaseTypeList(algorithmBaseType);
        ExcelUtil<AlgorithmBaseType> util = new ExcelUtil<AlgorithmBaseType>(AlgorithmBaseType.class);
        util.exportExcel(response, list, "算法预警类型数据");
    }

    /**
     * 获取算法预警类型详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:type:query')")
    @GetMapping(value = "/{algorithmId}")
    public AjaxResult getInfo(@PathVariable("algorithmId") Long algorithmId)
    {
        return success(algorithmBaseTypeService.selectAlgorithmBaseTypeByAlgorithmId(algorithmId));
    }

    /**
     * 新增算法预警类型
     */
    @PreAuthorize("@ss.hasPermi('system:type:add')")
    @Log(title = "算法预警类型", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody AlgorithmBaseType algorithmBaseType)
    {
        return toAjax(algorithmBaseTypeService.insertAlgorithmBaseType(algorithmBaseType));
    }

    /**
     * 修改算法预警类型
     */
    @PreAuthorize("@ss.hasPermi('system:type:edit')")
    @Log(title = "算法预警类型", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody AlgorithmBaseType algorithmBaseType)
    {
        return toAjax(algorithmBaseTypeService.updateAlgorithmBaseType(algorithmBaseType));
    }

    /**
     * 删除算法预警类型
     */
    @PreAuthorize("@ss.hasPermi('system:type:remove')")
    @Log(title = "算法预警类型", businessType = BusinessType.DELETE)
	@DeleteMapping("/{algorithmIds}")
    public AjaxResult remove(@PathVariable Long[] algorithmIds)
    {
        return toAjax(algorithmBaseTypeService.deleteAlgorithmBaseTypeByAlgorithmIds(algorithmIds));
    }
}
