package com.censoft.flink;

import com.censoft.flink.domain.AlgorithmPushDto;
import com.censoft.flink.domain.AlgorithmSceneBasePo;
import com.censoft.flink.domain.AlgorithmScenePiecePo;
import com.censoft.flink.mapper.SqlFactory;
import com.censoft.flink.mqtt.MqttConsumer;
import com.censoft.flink.transform.AlgorithmBaseFilterFunction;
import com.censoft.flink.transform.AlgorithmTypeFlatMapFunction;
import com.censoft.flink.transform.ResultExistFilterFunction;
import com.censoft.flink.transform.UpdateLiveFilterFunction;
import org.apache.flink.api.java.utils.ParameterTool;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.streaming.api.datastream.DataStream;
import org.apache.flink.streaming.api.datastream.SingleOutputStreamOperator;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;


public class StreamingJob {

    public static void main(String[] args) throws Exception {
        //1.获取对应场景id
        ParameterTool parameterTool = ParameterTool.fromArgs(args);
        Long sceneId = parameterTool.getLong("sceneId");

        //2.获取对应算法参数
        //2.1 获取场景数据
        SqlFactory sqlFactory = new SqlFactory();
        AlgorithmSceneBasePo algorithmSceneBasePo = sqlFactory.getAlgorithmSceneBasePo(sceneId);

        //2.2 获取场景算法块数据
        List<AlgorithmScenePiecePo> algorithmScenePieceList = sqlFactory.getAlgorithmScenePieceList(sceneId);


        //1、创建流式执行环境
        StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment();
        env.setParallelism(1);


        //2、接收消息并将数据转化为流
        //根据场景id 拼接对应的mqtt频道名称
        DataStream<AlgorithmPushDto> mqttStream = env.addSource(new MqttConsumer("/censoft/cpptest/" + sceneId));

        //3、进行处理
        //3、0 更新场景在线时间
        SingleOutputStreamOperator<AlgorithmPushDto> streamOperator = mqttStream
                .filter(new UpdateLiveFilterFunction(sceneId));

        //3、1 默认 是否存在结果集判断
        streamOperator = streamOperator
                .filter(new ResultExistFilterFunction());

        //3、2 默认 根据分类,分解多个推送信息
        List<String> algorithmTypes = Arrays.asList(algorithmSceneBasePo.getAlarmTypes().split(","));
        streamOperator = streamOperator.flatMap(new AlgorithmTypeFlatMapFunction(algorithmTypes));


        //3、3 根据配置算法流程 反射对应对象
        for (AlgorithmScenePiecePo algorithmScenePiecePo : algorithmScenePieceList) {
            //通过构造函数 获取算法块对象
            Class<?> filterClazz = Class.forName(algorithmScenePiecePo.getClazz());
            Method method = filterClazz.getMethod("getFilterFunction", AlgorithmSceneBasePo.class, AlgorithmScenePiecePo.class);
            AlgorithmBaseFilterFunction filterFunction = (AlgorithmBaseFilterFunction) method.invoke(null, algorithmSceneBasePo, algorithmScenePiecePo);
            streamOperator = streamOperator.filter(filterFunction);
        }

        //3、4 Object -> String
        SingleOutputStreamOperator<String> outputStreamOperator = streamOperator.map(algorithmPushDto ->{
            System.out.println(algorithmPushDto);
            return algorithmPushDto.toString();
        });

        //3、5 输出kafka
//        outputStreamOperator.addSink(new FlinkKafkaProducer("192.168.10.137:9092", "test-topic", new SimpleStringSchema()));

        //3、6 打印输出
        outputStreamOperator.print();
        outputStreamOperator.writeAsText("D:/word"+sceneId+".txt", FileSystem.WriteMode.OVERWRITE).setParallelism(1);
        //3、7 自动执行
        env.execute();
    }
}
