package com.censoft.flink.transform;

import com.censoft.flink.domain.AlgorithmPushDto;
import com.censoft.flink.domain.AlgorithmSceneBasePo;
import com.censoft.flink.domain.AlgorithmScenePiecePo;
import com.censoft.flink.domain.AlgorithmScenePieceVariablePo;
import com.censoft.flink.exception.ParameterTransformException;

import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author 周昊
 * @desc 报警间隔计算筛选
 */
public class AlarmSecondFilterFunction implements AlgorithmBaseFilterFunction {

    //alarmSecond 对应报警间隔 毫秒
    private final Long alarmSecond;

    //用来存储报警间隔算法缓存对象
    private final ConcurrentHashMap<String, Long> cache = new ConcurrentHashMap();

    //对应报警间隔 参数KEY
    private static String alarmSecondKey = "alarm_second";

    public AlarmSecondFilterFunction(Long alarmSecond) {
        this.alarmSecond = alarmSecond;
    }

    public static AlarmSecondFilterFunction getFilterFunction(AlgorithmSceneBasePo algorithmSceneBasePo, AlgorithmScenePiecePo algorithmScenePiecePo) {
        //参数准备
        Optional<AlgorithmScenePieceVariablePo> variablePo = algorithmScenePiecePo.getVariablePos()
                .stream()
                .filter(po -> alarmSecondKey.equals(po.getVariableKey()))
                .findFirst();

        //判断参数是否存在，如果不存在抛出异常
        if (!variablePo.isPresent()) {
            throw new ParameterTransformException();
        }
        Long alarmSecond = Long.valueOf(variablePo.get().getVariableValue());

        return new AlarmSecondFilterFunction(alarmSecond);
    }

    @Override
    public boolean filter(AlgorithmPushDto algorithmPushDto) {
        String algorithmType = algorithmPushDto.getAlgorithmType() + "_" + algorithmPushDto.getSort();

        Long alarmTimeStamp = cache.get(algorithmType);

        if (alarmTimeStamp == null) {
            alarmTimeStamp = 0L;
            cache.put(algorithmType, alarmTimeStamp);
        }

        //判断是否超过报警间隔
        if (alarmTimeStamp + alarmSecond * 1000 <= algorithmPushDto.getTimeStamp()) {
            cache.put(algorithmType, algorithmPushDto.getTimeStamp());
            return true;
        }
        return false;
    }
}
