package com.censoft.flink.transform;

import com.censoft.flink.domain.*;
import com.censoft.flink.exception.ParameterTransformException;
import org.apache.log4j.LogManager;
import org.apache.log4j.Logger;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author 周昊
 * @desc 帧计算筛选
 */
public class FrameFilterFunction implements AlgorithmBaseFilterFunction {

    //frameSecondKey 每秒几帧
    private final Integer frameSecond;

    //frameNumberKey 连续多少帧算报警
    private final Long frameNumber;

    //用来存储帧算法缓存对象
    private final ConcurrentHashMap<String, AlgorithmFrameNumberDto> cache = new ConcurrentHashMap();

    //连续多少帧算报警 参数KEY
    private static String frameNumberKey = "frame";


    public FrameFilterFunction(Integer frameSecond, Long frameNumber) {
        this.frameSecond = frameSecond;
        this.frameNumber = frameNumber;
    }

    public static AlgorithmBaseFilterFunction getFilterFunction(AlgorithmSceneBasePo algorithmSceneBasePo, AlgorithmScenePiecePo algorithmScenePiecePo) {
        //参数准备
        Optional<AlgorithmScenePieceVariablePo> variablePo = algorithmScenePiecePo.getVariablePos()
                .stream()
                .filter(po -> frameNumberKey.equals(po.getVariableKey()))
                .findFirst();

        //判断参数是否存在，如果不存在抛出异常
        if (algorithmSceneBasePo.getFrameSecond() == null || !variablePo.isPresent()) {
            throw new ParameterTransformException();
        }


        Integer frameSecond = algorithmSceneBasePo.getFrameSecond();
        Long frameNumber = Long.valueOf(variablePo.get().getVariableValue());

        return new FrameFilterFunction(frameSecond, frameNumber);
    }

    @Override
    public boolean filter(AlgorithmPushDto algorithmPushDto) {
        String algorithmType = algorithmPushDto.getAlgorithmType() + "_" + algorithmPushDto.getSort();

        AlgorithmFrameNumberDto frameNumberDto = cache.get(algorithmType);

        if (frameNumberDto == null) {
            frameNumberDto = new AlgorithmFrameNumberDto();
            frameNumberDto.setFrameNumber(0);
            frameNumberDto.setTimeStamp(0L);
            cache.put(algorithmType, frameNumberDto);
        }

        //获取对应帧数匹配规则
        //3.判断是否符合对应帧数限制
        //如果为连续 redis计数器加一 不连续制空
        if (frameNumberDto.getTimeStamp() * 1000 + 1000 / frameSecond >= algorithmPushDto.getTimeStamp() * 1000) {
            frameNumberDto.setTimeStamp(algorithmPushDto.getTimeStamp()* 1000);
            frameNumberDto.setFrameNumber(frameNumberDto.getFrameNumber() + 1);
            cache.put(algorithmType, frameNumberDto);

            //连续且超过连续报警帧数 下一步 ，反之退出
            if (frameNumberDto.getFrameNumber() < frameNumber) {
                return false;
            }
        } else {
            frameNumberDto.setTimeStamp(algorithmPushDto.getTimeStamp() *1000);
            frameNumberDto.setFrameNumber(0);
            cache.put(algorithmType, frameNumberDto);
            return false;
        }

        return true;
    }
}
